/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.registry;

import de.fraunhofer.iese.ind2uce.api.component.Component;
import de.fraunhofer.iese.ind2uce.api.component.description.MethodInterfaceDescription;
import de.fraunhofer.iese.ind2uce.api.component.identifier.ComponentId;
import de.fraunhofer.iese.ind2uce.api.component.interfaces.IPolicyManagementPoint;
import de.fraunhofer.iese.ind2uce.logger.LoggerFactory;
import org.slf4j.Logger;

import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * Builder to register a component (pxp, pip) at the Management Point.
 * Usage with fluent api:
 *
 * <pre class="code">
 *     InterfaceDescription d = ..;
 *     RegistryBuilder builder = new RegistryBuilder(ComponentType.PIP,pmp);
 *     boolean isRegistered = builder.componentId(new ComponentId("urn:component:test:pip:1234"))
 *      .addInterfaceDescription(d)
 *      .addUri("http://localhost:8008")
 *      .register();
 * </pre>
 */
public class RegistryBuilder {

  /**
   * The Constant LOG.
   */
  private static final Logger LOG = LoggerFactory.getLogger(RegistryBuilder.class);

  /**
   * The component type.
   */
  private final ComponentType componentType;

  /**
   * The pmp.
   */
  private final IPolicyManagementPoint pmp;

  /**
   * The urls.
   */
  private final List<URI> urls = new ArrayList<>();

  /**
   * The interface descriptions.
   */
  private final List<MethodInterfaceDescription> interfaceDescriptions = new ArrayList<>();

  /**
   * The to register.
   */
  private Component toRegister;

  /**
   * The component id.
   */
  private ComponentId componentId;

  /**
   * Instantiates a new registry builder.
   *
   * @param componentType the component type
   * @param registration the registration
   */
  public RegistryBuilder(ComponentType componentType, IPolicyManagementPoint registration) {
    this.componentType = componentType;
    this.pmp = registration;
  }

  /**
   * Component id.
   *
   * @param componentId the component id
   * @return the registry builder
   */
  public RegistryBuilder componentId(String componentId) {
    this.componentId = new ComponentId(this.componentType.getComponentIdPrefix() + ":" + componentId);
    return this;
  }

  /**
   * Component id.
   *
   * @param componentId the component id
   * @return the registry builder
   */
  public RegistryBuilder componentId(ComponentId componentId) {
    this.componentId = componentId;
    return this;
  }

  /**
   * Adds the interface description.
   *
   * @param description the description
   * @return the registry builder
   */
  public RegistryBuilder addInterfaceDescription(MethodInterfaceDescription description) {
    this.interfaceDescriptions.add(description);
    return this;
  }

  /**
   * Adds the all.
   *
   * @param interfaceDescriptions the interface descriptions
   * @return the registry builder
   */
  public RegistryBuilder addAll(Set<MethodInterfaceDescription> interfaceDescriptions) {
    this.interfaceDescriptions.addAll(interfaceDescriptions);
    return this;
  }

  /**
   * Adds the uri.
   *
   * @param uri the uri
   * @return the registry builder
   */
  public RegistryBuilder addUri(String uri) {
    this.urls.add(URI.create(uri));
    return this;
  }

  /**
   * Adds the uri.
   *
   * @param uri the uri
   * @return the registry builder
   */
  public RegistryBuilder addUri(URI uri) {
    this.urls.add(uri);
    return this;
  }

  /**
   * Register.
   *
   * @return true, if successful
   * @throws IOException Signals that an I/O exception has occurred.
   */
  public boolean register() throws IOException {
    LOG.info("RegistrationProcess started..");
    this.toRegister = new Component(this.componentId);

    this.toRegister.setUrls(this.urls);
    this.toRegister.setComponentInterfaces(this.interfaceDescriptions);

    final boolean registrationStatus = this.pmp.registerComponent(this.toRegister);

    if (registrationStatus) {
      LOG.info("Component {} successfully registered..", this.componentId);
    } else {
      LOG.warn("Component {} not registered", this.componentId);
    }
    return registrationStatus;
  }

}
