/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.pep.modifiermethods;

import de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterList;
import de.fraunhofer.iese.ind2uce.pep.common.ModifierMethod;
import de.fraunhofer.iese.ind2uce.registry.ActionDescription;
import de.fraunhofer.iese.ind2uce.registry.ActionParameterDescription;

import com.jayway.jsonpath.DocumentContext;
import com.jayway.jsonpath.MapFunction;

/****
 * This a default Modifier method with PEP sdk which appends the prefix and
 * suffix to string where event parameter is either a Json Object or string
 * (primitive type)
 */
public class AppendModifierMethod implements ModifierMethod {
  private static final String name = "append";

  @ActionDescription(description = "Appends a prefix and/or suffix to an event parameter value", pepSupportedType = String.class)
  public DocumentContext append(DocumentContext documentContext, String expression,
      @ActionParameterDescription(name = "suffix", description = "String to append at the end", mandatory = false, type = java.lang.String.class) String suffix,
      @ActionParameterDescription(name = "prefix", description = "String to append at the head", mandatory = false, type = java.lang.String.class) String prefix) {
    final MapFunction append = (currentValue, configuration) -> {
      final StringBuilder builder = new StringBuilder();
      if (prefix != null) {
        builder.append(prefix);
      }
      builder.append(currentValue.toString());
      if (suffix != null) {
        builder.append(suffix);
      }
      return builder.toString();
    };
    return documentContext.map(expression, append);
  }

  @Override
  public DocumentContext doModification(DocumentContext documentContext, String expression, ParameterList modifierMethodParameterList) {
    final Object suffix = modifierMethodParameterList.getParameterValueForName("suffix");
    final String suffixParam = suffix != null ? suffix.toString() : null;
    final Object prefix = modifierMethodParameterList.getParameterValueForName("prefix");
    final String prefixParam = prefix != null ? prefix.toString() : null;
    return this.append(documentContext, expression, suffixParam, prefixParam);
  }

  @Override
  public Object doModification(Object currentObject, ParameterList modifierMethodParameterList) {

    final Object suffix = modifierMethodParameterList.getParameterValueForName("suffix");
    final String suffixParam = suffix != null ? suffix.toString() : null;
    final Object prefix = modifierMethodParameterList.getParameterValueForName("prefix");
    final String prefixParam = prefix != null ? prefix.toString() : null;

    final StringBuilder builder = new StringBuilder();
    if (prefixParam != null) {
      builder.append(prefixParam);
    }
    builder.append(currentObject.toString());
    if (suffixParam != null) {
      builder.append(suffixParam);
    }
    return builder.toString();
  }

  @Override
  public String getDisplayName() {
    return name;
  }
}
