/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.pep;

import de.fraunhofer.iese.ind2uce.api.component.PepComponent;
import de.fraunhofer.iese.ind2uce.api.component.description.ModifierInterfaceDescription;
import de.fraunhofer.iese.ind2uce.api.component.description.PepInterfaceDescription;
import de.fraunhofer.iese.ind2uce.api.component.identifier.ComponentId;
import de.fraunhofer.iese.ind2uce.api.component.interfaces.IPolicyManagementPoint;
import de.fraunhofer.iese.ind2uce.connectors.ConnectorFactory;
import de.fraunhofer.iese.ind2uce.connectors.OAuthCredentials;
import de.fraunhofer.iese.ind2uce.pep.common.DecisionEnforcer;

import java.io.IOException;
import java.net.URI;
import java.util.List;

/**
 * Should be instantiated to create PEP enable application
 */
public class PolicyEnforcementPoint extends DefaultPolicyEnforcementPoint {

  private static final long serialVersionUID = -3408886626406980250L;

  private final ComponentId id;

  private final List<PepInterfaceDescription> pepInterfaceDescriptions;

  private final List<ModifierInterfaceDescription> modifierInterfaceDescriptions;

  private final URI pepUrl;

  private PepComponent pepComponent;

  private IPolicyManagementPoint pmp;

  /**
   * Constructor. Initializes the Connection to PMP and registers the PEP.
   *
   * @param decisionEnforcer Enforcer to use.
   * @param id Id of the PEP.
   * @param pmpUrl URL to PMP.
   * @param pepInterfaceDescriptions Modifier this PEP exports.
   * @param modifierInterfaceDescriptions Modifiers Interface Description.
   * @param pepUrl URL to PEP.
   * @param initialize initialize?
   * @param credentials OAuthCredentials used to connect to PDP
   * @throws IOException {@link IOException}
   */
  public PolicyEnforcementPoint(DecisionEnforcer decisionEnforcer, ComponentId id, URI pmpUrl, List<PepInterfaceDescription> pepInterfaceDescriptions,
      List<ModifierInterfaceDescription> modifierInterfaceDescriptions, URI pepUrl, boolean initialize, OAuthCredentials credentials) throws IOException {
    super(pmpUrl, decisionEnforcer,credentials);
    this.id = id;
    this.pepUrl = pepUrl;
    this.pepInterfaceDescriptions = pepInterfaceDescriptions;
    this.modifierInterfaceDescriptions = modifierInterfaceDescriptions;
    if (initialize) {
      this.initialize();
    }
  }

  /**
   * Returns PEP componet ID
   *
   * @return The id of the component.
   * @throws IOException {@link IOException}
   */
  @Override
  public ComponentId getId() throws IOException {
    return this.id;
  }

  /**
   * initialization using registry builder
   *
   * @param args not used here.
   * @return true if PEP could be initialized.
   * @throws IOException If connection to PMP could not be established.
   */
  @Override
  public boolean initialize(String... args) throws IOException {
    super.initPpdConnection();
    this.pmp = ConnectorFactory.getPmpClient(this.pmpUrl,this.oauthClientCredentials);
    this.pepComponent = new PepComponent(this.id);
    this.pepComponent.addUrl(this.pepUrl != null ? this.pepUrl : null);
    this.pepComponent.setInterfaceDescriptions(this.pepInterfaceDescriptions);
    this.pepComponent.setModifierDescriptionList(this.modifierInterfaceDescriptions);
    final boolean registrationStatus = this.pmp.registerPEPComponent(this.pepComponent);
    if (Boolean.valueOf(registrationStatus) == true) {
      LOG.info("PEP initialization is done!");
    } else {
      LOG.warn("an error occurred while initializing PEP component");
    }
    return registrationStatus;
  }

}
