/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.logger;

import org.slf4j.Logger;
import org.slf4j.helpers.NOPLogger;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

/**
 * A factory for creating Logger objects.
 */
public class LoggerFactory {

  /**
   * The Enum LoggerFramework.
   */
  public enum LoggerFramework {

    /**
     * LOGBACK Logger.
     */
    LOGBACK,
    /**
     * SLF4J Logger.
     */
    SLF4J,
    /**
     * ANDROID Logger.
     */
    ANDROID,
    /**
     * STANDARD OUT Logger
     */
    STDOUT;
  }

  /**
   * The Enum LogLevels.
   */
  public enum LogLevel {

    /**
     * The trace.
     */
    TRACE,
    /**
     * The debug.
     */
    DEBUG,
    /**
     * The info.
     */
    INFO,
    /**
     * The warn.
     */
    WARN,
    /**
     * The error.
     */
    ERROR,
    /**
     * The off.
     */
    OFF;
  }

  /**
   * The used logger.
   */
  private static LoggerFramework usedLogger = LoggerFramework.LOGBACK;

  /**
   * The current log level.
   */
  private static LogLevel logLevel = LogLevel.DEBUG;

  /**
   * The android logger.
   */
  private static AndroidLogger androidLogger;

  /**
   * Gets the logger.
   *
   * @param clazz the clazz
   * @return the logger
   */
  public static Logger getLogger(Class<?> clazz) {
    return getLogger(clazz.getName());
  }

  /**
   * Gets the logger.
   *
   * @param name the name
   * @return the logger
   */
  public static Logger getLogger(String name) {
    try {
      final Class<?> loggerFactoryClass = Class.forName("org.slf4j.LoggerFactory");
      final Method method = loggerFactoryClass.getMethod("getLogger", String.class);
      final Object o = method.invoke(null, name);
      LoggerFactory.usedLogger = LoggerFramework.LOGBACK;
      if (!(o instanceof NOPLogger)) {
        return (Logger)o;
      }

    } catch (ClassNotFoundException | NoSuchMethodException | SecurityException | IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
      e.printStackTrace();// NOSONAR
    }

    try {
      @SuppressWarnings("unused")
      final Class<?> loggerFactoryClass = Class.forName("android.util.Log");
      LoggerFactory.usedLogger = LoggerFramework.ANDROID;
      androidLogger = new AndroidLogger(name, logLevel);
      return androidLogger;
    } catch (ClassNotFoundException | SecurityException | IllegalArgumentException e) {
      // do nothing on purpose
      e.printStackTrace();// NOSONAR
    }

    LoggerFactory.usedLogger = LoggerFramework.STDOUT;
    return new SysoutLogger(name);
  }

  /**
   * Sets the root level.
   *
   * @param newLevel the new level
   * @return true, if successful
   */
  public static boolean setRootLevel(LogLevel newLevel) {
    logLevel = newLevel;
    if (usedLogger == LoggerFramework.ANDROID) {
      androidLogger.setLogLevel(newLevel);
      return true;
    } else if (usedLogger == LoggerFramework.LOGBACK) {

      try {
        final Class<?> rootLogger = Class.forName("ch.qos.logback.classic.Logger");
        final Class<?> slf4jLoggerFactory = Class.forName("org.slf4j.LoggerFactory");

        final Method method = slf4jLoggerFactory.getMethod("getLogger", String.class);
        final Object o = method.invoke(null, "ROOT");

        final Class<?> levelClass = Class.forName("ch.qos.logback.classic.Level");
        final Method m2 = rootLogger.getMethod("setLevel", levelClass);
        final Method m3 = levelClass.getMethod("toLevel", int.class);

        switch (newLevel) {
          case OFF:
            m2.invoke(rootLogger.cast(o), m3.invoke(null, Integer.MAX_VALUE));
            break;
          case ERROR:
            m2.invoke(rootLogger.cast(o), m3.invoke(null, 40000));
            break;
          case WARN:
            m2.invoke(rootLogger.cast(o), m3.invoke(null, 30000));
            break;
          case INFO:
            m2.invoke(rootLogger.cast(o), m3.invoke(null, 20000));
            break;
          case DEBUG:
            m2.invoke(rootLogger.cast(o), m3.invoke(null, 10000));
            break;
          case TRACE:
            m2.invoke(rootLogger.cast(o), m3.invoke(null, 5000));
            break;
          default:
            m2.invoke(rootLogger.cast(o), m3.invoke(null, 10000));
            break;
        }
      } catch (ClassNotFoundException | IllegalAccessException | IllegalArgumentException | InvocationTargetException | NoSuchMethodException | SecurityException e) {
        e.printStackTrace();// NOSONAR
        return false;
      }
    } else {
      System.err.println("LoggerFramework not supported");// NOSONAR
      return false;
    }
    return true;
  }
}
