/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.logger;

import de.fraunhofer.iese.ind2uce.logger.LoggerFactory.LogLevel;

import org.slf4j.Logger;
import org.slf4j.Marker;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

/**
 * The Class AndroidLogger.
 */
public class AndroidLogger implements Logger {

  /**
   * The placeholder used by String.format.
   */
  private static final String STRING_FORMAT = "%s";

  /**
   * The placeholders used by logback.
   */
  private static final String REGEX = "\\{\\}";

  /**
   * The Constant TRACE_LEVEL.
   */
  private static final int TRACE_LEVEL = 5000;

  /**
   * The Constant DEBUG_LEVEL.
   */
  private static final int DEBUG_LEVEL = 10000;

  /**
   * The Constant INFO_LEVEL.
   */
  private static final int INFO_LEVEL = 20000;

  /**
   * The Constant WARN_LEVEL.
   */
  private static final int WARN_LEVEL = 30000;

  /**
   * The Constant ERROR_LEVEL.
   */
  private static final int ERROR_LEVEL = 40000;

  /**
   * The Constant OFF_LEVEL.
   */
  private static final int OFF_LEVEL = Integer.MAX_VALUE;

  /**
   * The name.
   */
  private String name = "AndroidLogger";

  /**
   * The android logger.
   */
  private Class<?> androidLoggerClass;

  /**
   * The trace.
   */
  private Method trace;

  /**
   * The debug.
   */
  private Method debug;

  /**
   * The info.
   */
  private Method info;

  /**
   * The warn.
   */
  private Method warn;

  /**
   * The error.
   */
  private Method error;

  /**
   * The log level.
   */
  private LogLevel logLevel;

  /**
   * Instantiates a new android logger.
   *
   * @param name the name
   * @param logLevel the log level
   */
  public AndroidLogger(String name, LogLevel logLevel) {
    this.name = name;
    this.logLevel = logLevel;
    try {
      this.androidLoggerClass = Class.forName("android.util.Log");
      this.trace = this.androidLoggerClass.getMethod("v", String.class, String.class);
      this.debug = this.androidLoggerClass.getMethod("d", String.class, String.class);
      this.info = this.androidLoggerClass.getMethod("i", String.class, String.class);
      this.warn = this.androidLoggerClass.getMethod("w", String.class, String.class);
      this.error = this.androidLoggerClass.getMethod("e", String.class, String.class);
    } catch (ClassNotFoundException | NoSuchMethodException | SecurityException | IllegalArgumentException e) {
      System.err.println("androidLoggerClass not found!"); // NOSONAR
      e.printStackTrace();// NOSONAR
    }
  }

  /**
   * Check log level.
   *
   * @param logLevel the log level
   * @return true, if successful
   */
  private boolean checkLogLevel(LogLevel logLevel) {
    return (this.getLogLevelValue(this.logLevel) <= this.getLogLevelValue(logLevel));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#debug(org.slf4j.Marker, java.lang.String)
   */
  @Override
  public void debug(Marker marker, String msg) {
    try {
      this.debug.invoke(null, this.shortenName(this.name), msg);
    } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
      e.printStackTrace();// NOSONAR
    }
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#debug(org.slf4j.Marker, java.lang.String,
   * java.lang.Object)
   */
  @Override
  public void debug(Marker marker, String format, Object arg) {
    this.debug(String.format(format.replaceAll(REGEX, STRING_FORMAT), arg));

  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#debug(org.slf4j.Marker, java.lang.String,
   * java.lang.Object[])
   */
  @Override
  public void debug(Marker marker, String format, Object... arguments) {
    this.debug(String.format(format.replaceAll(REGEX, STRING_FORMAT), arguments));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#debug(org.slf4j.Marker, java.lang.String,
   * java.lang.Object, java.lang.Object)
   */
  @Override
  public void debug(Marker marker, String format, Object arg1, Object arg2) {
    this.debug(String.format(format.replaceAll(REGEX, STRING_FORMAT), arg1, arg2));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#debug(org.slf4j.Marker, java.lang.String,
   * java.lang.Throwable)
   */
  @Override
  public void debug(Marker marker, String msg, Throwable t) {
    this.debug(msg + "\n" + this.stackTraceToString(t));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#debug(java.lang.String)
   */
  @Override
  public void debug(String msg) {
    try {
      if (this.checkLogLevel(LogLevel.DEBUG)) {
        this.debug.invoke(null, this.shortenName(this.name), msg);
      }
    } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
      e.printStackTrace();// NOSONAR
    }
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#debug(java.lang.String, java.lang.Object)
   */
  @Override
  public void debug(String format, Object arg) {
    this.debug(String.format(format.replaceAll(REGEX, STRING_FORMAT), arg));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#debug(java.lang.String, java.lang.Object[])
   */
  @Override
  public void debug(String format, Object... arguments) {
    this.debug(String.format(format.replaceAll(REGEX, STRING_FORMAT), arguments));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#debug(java.lang.String, java.lang.Object,
   * java.lang.Object)
   */
  @Override
  public void debug(String format, Object arg1, Object arg2) {
    this.debug(String.format(format.replaceAll(REGEX, STRING_FORMAT), arg1, arg2));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#debug(java.lang.String, java.lang.Throwable)
   */
  @Override
  public void debug(String msg, Throwable t) {
    this.debug(msg + "\n" + this.stackTraceToString(t));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#error(org.slf4j.Marker, java.lang.String)
   */
  @Override
  public void error(Marker marker, String msg) {
    try {
      this.error.invoke(null, this.shortenName(this.name), msg);
    } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
      e.printStackTrace();// NOSONAR
    }
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#error(org.slf4j.Marker, java.lang.String,
   * java.lang.Object)
   */
  @Override
  public void error(Marker marker, String format, Object arg) {
    this.error(String.format(format.replaceAll(REGEX, STRING_FORMAT), arg));

  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#error(org.slf4j.Marker, java.lang.String,
   * java.lang.Object[])
   */
  @Override
  public void error(Marker marker, String format, Object... arguments) {
    this.error(String.format(format.replaceAll(REGEX, STRING_FORMAT), arguments));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#error(org.slf4j.Marker, java.lang.String,
   * java.lang.Object, java.lang.Object)
   */
  @Override
  public void error(Marker marker, String format, Object arg1, Object arg2) {
    this.error(String.format(format.replaceAll(REGEX, STRING_FORMAT), arg1, arg2));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#error(org.slf4j.Marker, java.lang.String,
   * java.lang.Throwable)
   */
  @Override
  public void error(Marker marker, String msg, Throwable t) {
    this.error(msg + "\n" + this.stackTraceToString(t));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#error(java.lang.String)
   */
  @Override
  public void error(String msg) {
    try {
      if (this.checkLogLevel(LogLevel.ERROR)) {
        this.error.invoke(null, this.shortenName(this.name), msg);
      }
    } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
      e.printStackTrace();// NOSONAR
    }
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#error(java.lang.String, java.lang.Object)
   */
  @Override
  public void error(String format, Object arg) {
    this.error(String.format(format.replaceAll(REGEX, STRING_FORMAT), arg));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#error(java.lang.String, java.lang.Object[])
   */
  @Override
  public void error(String format, Object... arguments) {
    this.error(String.format(format.replaceAll(REGEX, STRING_FORMAT), arguments));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#error(java.lang.String, java.lang.Object,
   * java.lang.Object)
   */
  @Override
  public void error(String format, Object arg1, Object arg2) {
    this.error(String.format(format.replaceAll(REGEX, STRING_FORMAT), arg1, arg2));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#error(java.lang.String, java.lang.Throwable)
   */
  @Override
  public void error(String msg, Throwable t) {
    this.error(msg + "\n" + this.stackTraceToString(t));
  }

  /**
   * Gets the log level.
   *
   * @return the log level
   */
  public LogLevel getLogLevel() {
    return this.logLevel;
  }

  /**
   * Gets the log level value.
   *
   * @param logLevel the log level
   * @return the log level value
   */
  private int getLogLevelValue(LogLevel logLevel) {
    switch (logLevel) {
      case OFF:
        return OFF_LEVEL;
      case TRACE:
        return TRACE_LEVEL;
      case DEBUG:
        return DEBUG_LEVEL;
      case INFO:
        return INFO_LEVEL;
      case WARN:
        return WARN_LEVEL;
      case ERROR:
        return ERROR_LEVEL;
      default:
        return DEBUG_LEVEL;
    }
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#getName()
   */
  @Override
  public String getName() {
    return this.name;
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#info(org.slf4j.Marker, java.lang.String)
   */
  @Override
  public void info(Marker marker, String msg) {
    try {
      this.info.invoke(null, this.shortenName(this.name), msg);
    } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
      e.printStackTrace();// NOSONAR
    }
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#info(org.slf4j.Marker, java.lang.String,
   * java.lang.Object)
   */
  @Override
  public void info(Marker marker, String format, Object arg) {
    this.info(String.format(format.replaceAll(REGEX, STRING_FORMAT), arg));

  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#info(org.slf4j.Marker, java.lang.String,
   * java.lang.Object[])
   */
  @Override
  public void info(Marker marker, String format, Object... arguments) {
    this.info(String.format(format.replaceAll(REGEX, STRING_FORMAT), arguments));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#info(org.slf4j.Marker, java.lang.String,
   * java.lang.Object, java.lang.Object)
   */
  @Override
  public void info(Marker marker, String format, Object arg1, Object arg2) {
    this.info(String.format(format.replaceAll(REGEX, STRING_FORMAT), arg1, arg2));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#info(org.slf4j.Marker, java.lang.String,
   * java.lang.Throwable)
   */
  @Override
  public void info(Marker marker, String msg, Throwable t) {
    this.info(msg + "\n" + this.stackTraceToString(t));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#info(java.lang.String)
   */
  @Override
  public void info(String msg) {
    try {
      if (this.checkLogLevel(LogLevel.INFO)) {
        this.info.invoke(null, this.shortenName(this.name), msg);
      }
    } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
      e.printStackTrace();// NOSONAR
    }
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#info(java.lang.String, java.lang.Object)
   */
  @Override
  public void info(String format, Object arg) {
    this.info(String.format(format.replaceAll(REGEX, STRING_FORMAT), arg));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#info(java.lang.String, java.lang.Object[])
   */
  @Override
  public void info(String format, Object... arguments) {
    this.info(String.format(format.replaceAll(REGEX, STRING_FORMAT), arguments));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#info(java.lang.String, java.lang.Object,
   * java.lang.Object)
   */
  @Override
  public void info(String format, Object arg1, Object arg2) {
    this.info(String.format(format.replaceAll(REGEX, STRING_FORMAT), arg1, arg2));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#info(java.lang.String, java.lang.Throwable)
   */
  @Override
  public void info(String msg, Throwable t) {
    this.info(msg + "\n" + this.stackTraceToString(t));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#isDebugEnabled()
   */
  @Override
  public boolean isDebugEnabled() {
    return this.checkLogLevel(LogLevel.DEBUG);
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#isDebugEnabled(org.slf4j.Marker)
   */
  @Override
  public boolean isDebugEnabled(Marker marker) {
    return this.checkLogLevel(LogLevel.DEBUG);
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#isErrorEnabled()
   */
  @Override
  public boolean isErrorEnabled() {
    return this.checkLogLevel(LogLevel.ERROR);
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#isErrorEnabled(org.slf4j.Marker)
   */
  @Override
  public boolean isErrorEnabled(Marker marker) {
    return this.checkLogLevel(LogLevel.ERROR);
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#isInfoEnabled()
   */
  @Override
  public boolean isInfoEnabled() {
    return this.checkLogLevel(LogLevel.INFO);
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#isInfoEnabled(org.slf4j.Marker)
   */
  @Override
  public boolean isInfoEnabled(Marker marker) {
    return this.checkLogLevel(LogLevel.INFO);
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#isTraceEnabled()
   */
  @Override
  public boolean isTraceEnabled() {
    return this.checkLogLevel(LogLevel.TRACE);
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#isTraceEnabled(org.slf4j.Marker)
   */
  @Override
  public boolean isTraceEnabled(Marker marker) {
    return this.checkLogLevel(LogLevel.TRACE);
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#isWarnEnabled()
   */
  @Override
  public boolean isWarnEnabled() {
    return this.checkLogLevel(LogLevel.WARN);
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#isWarnEnabled(org.slf4j.Marker)
   */
  @Override
  public boolean isWarnEnabled(Marker marker) {
    return this.checkLogLevel(LogLevel.WARN);
  }

  /**
   * Sets the log level.
   *
   * @param logLevel the new log level
   */
  public void setLogLevel(LogLevel logLevel) {
    this.logLevel = logLevel;
  }

  /**
   * Shorten name.
   *
   * @param name the name
   * @return the string
   */
  private String shortenName(String name) {
    if (name.length() > 30) {
      return "..." + name.substring(name.length() - 30, name.length());
    } else {
      return name;
    }
  }

  /**
   * Stack trace to string.
   *
   * @param e the exception or throwable
   * @return the stacktrace
   */
  private String stackTraceToString(Throwable e) {
    final StringWriter sw = new StringWriter();
    final PrintWriter pw = new PrintWriter(sw);
    e.printStackTrace(pw);
    return sw.toString();
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#trace(org.slf4j.Marker, java.lang.String)
   */
  @Override
  public void trace(Marker marker, String msg) {
    try {
      this.trace.invoke(null, this.shortenName(this.name), msg);
    } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
      e.printStackTrace();// NOSONAR
    }
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#trace(org.slf4j.Marker, java.lang.String,
   * java.lang.Object)
   */
  @Override
  public void trace(Marker marker, String format, Object arg) {
    this.trace(String.format(format.replaceAll(REGEX, STRING_FORMAT), arg));

  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#trace(org.slf4j.Marker, java.lang.String,
   * java.lang.Object[])
   */
  @Override
  public void trace(Marker marker, String format, Object... arguments) {
    this.trace(String.format(format.replaceAll(REGEX, STRING_FORMAT), arguments));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#trace(org.slf4j.Marker, java.lang.String,
   * java.lang.Object, java.lang.Object)
   */
  @Override
  public void trace(Marker marker, String format, Object arg1, Object arg2) {
    this.trace(String.format(format.replaceAll(REGEX, STRING_FORMAT), arg1, arg2));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#trace(org.slf4j.Marker, java.lang.String,
   * java.lang.Throwable)
   */
  @Override
  public void trace(Marker marker, String msg, Throwable t) {
    this.trace(msg + "\n" + this.stackTraceToString(t));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#trace(java.lang.String)
   */
  @Override
  public void trace(String msg) {
    try {
      if (this.checkLogLevel(LogLevel.TRACE)) {
        this.trace.invoke(null, this.shortenName(this.name), msg);
      }
    } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
      e.printStackTrace();// NOSONAR
    }
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#trace(java.lang.String, java.lang.Object)
   */
  @Override
  public void trace(String format, Object arg) {
    this.trace(String.format(format.replaceAll(REGEX, STRING_FORMAT), arg));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#trace(java.lang.String, java.lang.Object[])
   */
  @Override
  public void trace(String format, Object... arguments) {
    this.trace(String.format(format.replaceAll(REGEX, STRING_FORMAT), arguments));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#trace(java.lang.String, java.lang.Object,
   * java.lang.Object)
   */
  @Override
  public void trace(String format, Object arg1, Object arg2) {
    this.trace(String.format(format.replaceAll(REGEX, STRING_FORMAT), arg1, arg2));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#trace(java.lang.String, java.lang.Throwable)
   */
  @Override
  public void trace(String msg, Throwable t) {
    this.trace(msg + "\n" + this.stackTraceToString(t));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#warn(org.slf4j.Marker, java.lang.String)
   */
  @Override
  public void warn(Marker marker, String msg) {
    try {
      this.warn.invoke(null, this.shortenName(this.name), msg);
    } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
      e.printStackTrace();// NOSONAR
    }
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#warn(org.slf4j.Marker, java.lang.String,
   * java.lang.Object)
   */
  @Override
  public void warn(Marker marker, String format, Object arg) {
    this.warn(String.format(format.replaceAll(REGEX, STRING_FORMAT), arg));

  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#warn(org.slf4j.Marker, java.lang.String,
   * java.lang.Object[])
   */
  @Override
  public void warn(Marker marker, String format, Object... arguments) {
    this.warn(String.format(format.replaceAll(REGEX, STRING_FORMAT), arguments));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#warn(org.slf4j.Marker, java.lang.String,
   * java.lang.Object, java.lang.Object)
   */
  @Override
  public void warn(Marker marker, String format, Object arg1, Object arg2) {
    this.warn(String.format(format.replaceAll(REGEX, STRING_FORMAT), arg1, arg2));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#warn(org.slf4j.Marker, java.lang.String,
   * java.lang.Throwable)
   */
  @Override
  public void warn(Marker marker, String msg, Throwable t) {
    this.warn(msg + "\n" + this.stackTraceToString(t));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#warn(java.lang.String)
   */
  @Override
  public void warn(String msg) {
    try {
      if (this.checkLogLevel(LogLevel.WARN)) {
        this.warn.invoke(null, this.shortenName(this.name), msg);
      }
    } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
      e.printStackTrace();// NOSONAR
    }
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#warn(java.lang.String, java.lang.Object)
   */
  @Override
  public void warn(String format, Object arg) {
    this.warn(String.format(format.replaceAll(REGEX, STRING_FORMAT), arg));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#warn(java.lang.String, java.lang.Object[])
   */
  @Override
  public void warn(String format, Object... arguments) {
    this.warn(String.format(format.replaceAll(REGEX, STRING_FORMAT), arguments));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#warn(java.lang.String, java.lang.Object,
   * java.lang.Object)
   */
  @Override
  public void warn(String format, Object arg1, Object arg2) {
    this.warn(String.format(format.replaceAll(REGEX, STRING_FORMAT), arg1, arg2));
  }

  /*
   * (non-Javadoc)
   * @see org.slf4j.Logger#warn(java.lang.String, java.lang.Throwable)
   */
  @Override
  public void warn(String msg, Throwable t) {
    this.warn(msg + "\n" + this.stackTraceToString(t));
  }
}
