/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.connectors;

import de.fraunhofer.iese.ind2uce.api.component.Component;
import de.fraunhofer.iese.ind2uce.api.component.ComponentBase;
import de.fraunhofer.iese.ind2uce.api.component.ComponentType;
import de.fraunhofer.iese.ind2uce.api.component.PepComponent;
import de.fraunhofer.iese.ind2uce.api.component.interfaces.IComponent;
import de.fraunhofer.iese.ind2uce.api.component.interfaces.IPolicyDecisionPoint;
import de.fraunhofer.iese.ind2uce.api.component.interfaces.IPolicyEnforcementPoint;
import de.fraunhofer.iese.ind2uce.api.component.interfaces.IPolicyExecutionPoint;
import de.fraunhofer.iese.ind2uce.api.component.interfaces.IPolicyInformationPoint;
import de.fraunhofer.iese.ind2uce.api.component.interfaces.IPolicyManagementPoint;
import de.fraunhofer.iese.ind2uce.api.component.interfaces.IPolicyRetrievalPoint;
import de.fraunhofer.iese.ind2uce.api.component.interfaces.IRootPolicyManagementPoint;
import de.fraunhofer.iese.ind2uce.logger.LoggerFactory;

import org.reflections.Reflections;
import org.slf4j.Logger;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.net.URI;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

/**
 * Helper class for establishing connections to the IND²UCE components by using
 * a URI. It uses reflection to determine the type of the connector needed for a
 * certain protocol.
 */
public final class ConnectorFactory {

  /**
   * The logger instance.
   */
  private static final Logger LOG = LoggerFactory.getLogger(ConnectorFactory.class);

  /**
   * Priority in increasing order.
   */
  private static final List<String> connectorPriorities = Arrays.asList(new String[] {
      "binder", // Least priority as it will not work on non Android
                // devices, which have an own ConnectorFactory
      "amqp", // AMQP is good but quite special. That's why it is not preferred
              // in comparison with https.
      "http", // HTTP/REST interfaces are not so good wrt. performance
      "https", // HTTPS/REST interfaces are not so good wrt. performance
      "tcp", // Only available for PDP
      "rmi", // Least error prone and good wrt. performance
  });

  /**
   * Used as cache to hold all connector classes (annotated with Connector)
   * which are available in the jar.
   */
  private static Set<Class<?>> connectorClasses;

  /**
   * Private constructor to prevent instantiation of static class.
   */
  private ConnectorFactory() {

  }

  private static boolean contains(String[] array, String query) {
    for (final String element : array) {
      if (element.equals(query)) {
        return true;
      }
    }
    return false;
  }

  public static IComponent getComponent(final ComponentBase base, final String preferredConnectorType) {
    return getComponent(base, preferredConnectorType, null);
  }

  public static IComponent getComponent(final ComponentBase base, final String preferredConnectorType, Map<String, Authentication> protocolToAuthentication) {
    LOG.trace("Entering getConnector(base={}, preferredConnectorType={})", base, preferredConnectorType);
    if (null == base) {
      return null;
    }

    final IComponent component;
    switch (base.getType()) {
      case PDP:
        component = getPdp((Component)base, preferredConnectorType, protocolToAuthentication);
        break;

      case PEP:
        component = getPep((PepComponent)base, preferredConnectorType);
        break;

      case PMP:
        component = getPmpClient((Component)base, preferredConnectorType);
        break;

      case PMP_SERVER:
        component = getPmpServer((Component)base, preferredConnectorType);
        break;

      case PXP:
        component = getPxp((Component)base, preferredConnectorType);
        break;

      case PIP:
        component = getPip((Component)base, preferredConnectorType);
        break;

      case PRP:
        component = getPrp((Component)base, preferredConnectorType);
        break;

      default:
        LOG.warn("Unknown Component type {}", base.getType());
        component = null;
        break;
    }

    LOG.trace("Returning from getConnector(component={})", component);
    return component;
  }

  @SuppressWarnings("unchecked")
  /**
   * Uses reflection to lookup a connector class and instantiates it for a
   * certain communication protocol.
   *
   * @param component The component for which a connector should be requested.
   * @param type The type of the component requested.
   * @param preferredConnectorType The preferred connector type (optional). If
   *          null, if the preferredConnectorType string is not found or if
   *          there is only one url, the connector will be selected according to
   *          the priorities list. This parameter is not case sensitive.
   * @return A connector suitable for the specified protocol and component type,
   *         null in case of an error
   */
  private static <T extends IComponent> T getConnector(final ComponentBase component, final ComponentType type, final String preferredConnectorType,
      Map<String, Authentication> protocolToAuthentication) {
    LOG.info("Entering getConnector(component={}, type={})", component, type);

    if (component.getUrls() == null || component.getUrls().isEmpty()) {
      return null;
    }

    if (component.getUrls().size() == 1) {
      return getConnector(component.getUrls().get(0), type, null);
    }

    URI connectorURI = null;
    if (preferredConnectorType != null) {
      for (final URI uri : component.getUrls()) {
        if (uri.getScheme().equalsIgnoreCase(preferredConnectorType)) {
          connectorURI = uri;
          break;
        }
      }
    }

    if (connectorURI == null) {
      // Sort urls with respect to the defined priority
      Collections.sort(component.getUrls(), new Comparator<URI>() {
        @Override
        public int compare(URI o1, URI o2) {
          final Integer prio1 = this.getPriority(o1.getScheme());
          final Integer prio2 = this.getPriority(o2.getScheme());
          return -prio1.compareTo(prio2);
        }

        private int getPriority(String protocol) {
          return connectorPriorities.indexOf(protocol);
        }
      });

      connectorURI = component.getUrls().get(0);
    }
    Authentication authenticationForConnector = null;
    if (protocolToAuthentication != null && protocolToAuthentication.containsKey(connectorURI.getScheme())) {
      authenticationForConnector = protocolToAuthentication.get(connectorURI.getScheme());

    }
    final IComponent componentConnector = getConnector(connectorURI, type, authenticationForConnector);
    LOG.trace("Leaving getConnector(): {}", componentConnector);
    return (T)componentConnector;
  }

  @SuppressWarnings("unchecked")
  /**
   * Uses reflection to lookup a connector class and instantiates it for a
   * certain communication protocol.
   *
   * @param url The URL used to connect to the component, like
   *          rmi://localhost:1111/pmpRmi
   * @param type The type of the component requested.
   * @param credentials Credentials to login to pmp
   * @return A connector suitable for the specified protocol and component type,
   *         null in case of an error
   */
  private static <T extends IComponent> T getConnector(@Nonnull URI url, @Nonnull ComponentType type, @Nullable Authentication authentication) {
    LOG.trace("Entering getConnector(url={}, type={})", url, type);

    loadClasses();
    final String protocol = url.getScheme();
    String version = null;

    try {
      final Pattern p = Pattern.compile("(?<=version=).*?(?=&|$)");
      final Matcher m = p.matcher(url.toASCIIString());
      while (m.find()) {
        version = m.group();
      }
    } catch (final PatternSyntaxException ex) {
      LOG.warn("Could read version from scheme. Ignoring.", ex);
    }

    for (final Class<?> connector : connectorClasses) {
      final Connector annotation = connector.getAnnotation(Connector.class);
      final boolean annotationOk = annotation.type() == type;
      final boolean protocolOk = contains(annotation.protocol(), protocol);

      if (annotationOk && protocolOk) {
        try {

          final Constructor<?> constructor = getConstructor(connector, authentication);
          final T result = createInstance(url, authentication, constructor);
          LOG.trace("Leaving getConnector(): {}", result);
          return result;
        } catch (InstantiationException | IllegalAccessException | IllegalArgumentException | InvocationTargetException | NoSuchMethodException | SecurityException e) {
          LOG.warn("Could not instantiate connector", e);
        }
      }
    }

    LOG.info("Cannot find connector for (url={}, type={})", url, type);
    return null;
  }

  private static <T extends IComponent> T createInstance(URI url, Authentication credentials, Constructor<?> constructor)
      throws InstantiationException, IllegalAccessException, InvocationTargetException {
    return (T)(credentials != null ? constructor.newInstance(url, credentials) : constructor.newInstance(url));
  }

  private static Constructor<?> getConstructor(Class<?> connector, Authentication credentials) throws NoSuchMethodException {
    return credentials != null ? connector.getConstructor(URI.class, Authentication.class) : connector.getConstructor(URI.class);
  }

  /**
   * Tries to establish a connection to a PDP by using the passed url variable.
   *
   * @param component The Component containing the URL used to connect to the
   *          component, like rmi://localhost:1111/pdpRmi
   * @param preferredConnectorType The preferred connector type (optional). If
   *          null, if the preferredConnectorType string is not found or if
   *          there is only one url, the connector will be selected according to
   *          the priorities list. This parameter is not case sensitive.
   * @return A connector suitable for the specified protocol, null in case of an
   *         error
   */
  public static IPolicyDecisionPoint getPdp(Component component, final String preferredConnectorType, Map<String, Authentication> protocolToAuthentication) {
    return getConnector(component, ComponentType.PDP, preferredConnectorType, protocolToAuthentication);
  }

  /**
   * Tries to establish a connection to a PDP by using the passed url variable.
   *
   * @param url The URL used to connect to the component, like
   *          rmi://localhost:1111/pdpRmi
   * @return A connector suitable for the specified protocol, null in case of an
   *         error
   */
  public static IPolicyDecisionPoint getPdp(URI url) {
    return getConnector(url, ComponentType.PDP, null);
  }

  /**
   * Tries to establish a connection to a PDP by using the passed url variable.
   *
   * @param url The URL used to connect to the component, like
   *          rmi://localhost:1111/pdpRmi
   * @param credentials API Key (Oauth)
   * @return A connector suitable for the specified protocol, null in case of an
   *         error
   */
  public static IPolicyDecisionPoint getPdp(URI url, Authentication credentials) {
    return getConnector(url, ComponentType.PDP, credentials);
  }

  /**
   * Tries to establish a connection to a PEP by using the passed url variable.
   *
   * @param component The Component containing the URL used to connect to the
   *          component, like rmi://localhost:1111/pepRmi
   * @param preferredConnectorType The preferred connector type (optional). If
   *          null, if the preferredConnectorType string is not found or if
   *          there is only one url, the connector will be selected according to
   *          the priorities list. This parameter is not case sensitive.
   * @return A connector suitable for the specified protocol, null in case of an
   *         error
   */
  public static IPolicyEnforcementPoint getPep(PepComponent component, final String preferredConnectorType) {
    return getConnector(component, ComponentType.PEP, preferredConnectorType, null);
  }

  /**
   * Tries to establish a connection to a PEP by using the passed url variable.
   *
   * @param url The URL used to connect to the component, like
   *          rmi://localhost:1111/pepRmi
   * @return A connector suitable for the specified protocol, null in case of an
   *         error
   */
  public static IPolicyEnforcementPoint getPep(URI url) {
    return getConnector(url, ComponentType.PEP, null);
  }

  /**
   * Tries to establish a connection to a PIP by using the passed url variable.
   *
   * @param component The Component containing the URL used to connect to the
   *          component, like rmi://localhost:1111/pipRmi
   * @param preferredConnectorType The preferred connector type (optional). If
   *          null, if the preferredConnectorType string is not found or if
   *          there is only one url, the connector will be selected according to
   *          the priorities list. This parameter is not case sensitive.
   * @return A connector suitable for the specified protocol, null in case of an
   *         error
   */
  public static IPolicyInformationPoint getPip(Component component, final String preferredConnectorType) {
    return getConnector(component, ComponentType.PIP, preferredConnectorType, null);
  }

  /**
   * Tries to establish a connection to a PIP by using the passed url variable.
   *
   * @param url The URL used to connect to the component, like
   *          rmi://localhost:1111/pipRmi
   * @return A connector suitable for the specified protocol, null in case of an
   *         error
   */
  public static IPolicyInformationPoint getPip(URI url) {
    return getConnector(url, ComponentType.PIP, null);
  }

  /**
   * Tries to establish a connection to a PMP client by using the passed url
   * variable.
   *
   * @param component The Component containing the URL used to connect to the
   *          component, like rmi://localhost:1111/pmpRmi
   * @param preferredConnectorType The preferred connector type (optional). If
   *          null, if the preferredConnectorType string is not found or if
   *          there is only one url, the connector will be selected according to
   *          the priorities list. This parameter is not case sensitive.
   * @return A connector suitable for the specified protocol, null in case of an
   *         error
   */
  public static IPolicyManagementPoint getPmpClient(Component component, final String preferredConnectorType) {
    return getConnector(component, ComponentType.PMP, preferredConnectorType, null);
  }

  /**
   * Tries to establish a connection to a PMP client by using the passed url
   * variable.
   *
   * @param url The URL used to connect to the component, like
   *          rmi://localhost:1111/pmpRmi
   * @param oAuthCredentials Credentials to login.
   * @return A connector suitable for the specified protocol, null in case of an
   *         error
   */
  public static IPolicyManagementPoint getPmpClient(URI url, final OAuthCredentials oAuthCredentials) {
    return getConnector(url, ComponentType.PMP, oAuthCredentials);
  }

  /**
   * Tries to establish a connection to a PMP client by using the passed url
   * variable.
   *
   * @param url The URL used to connect to the component, like
   *          rmi://localhost:1111/pmpRmi
   * @return A connector suitable for the specified protocol, null in case of an
   *         error
   */
  public static IPolicyManagementPoint getPmpClient(URI url) {
    return getConnector(url, ComponentType.PMP, null);
  }

  /**
   * Tries to establish a connection to a PMP server by using the passed url
   * variable.
   *
   * @param component The Component containing the URL used to connect to the
   *          component, like rmi://localhost:1111/pmpRmi
   * @param preferredConnectorType The preferred connector type (optional). If
   *          null, if the preferredConnectorType string is not found or if
   *          there is only one url, the connector will be selected according to
   *          the priorities list. This parameter is not case sensitive.
   * @return A connector suitable for the specified protocol, null in case of an
   *         error
   */
  public static IRootPolicyManagementPoint getPmpServer(Component component, final String preferredConnectorType) {
    return getConnector(component, ComponentType.PMP_SERVER, preferredConnectorType, null);
  }

  /**
   * Tries to establish a connection to a PMP server by using the passed url
   * variable.
   *
   * @param url The URL used to connect to the component, like
   *          rmi://localhost:1111/pmpRmi
   * @return A connector suitable for the specified protocol, null in case of an
   *         error
   */
  public static IRootPolicyManagementPoint getPmpServer(URI url) {
    return getConnector(url, ComponentType.PMP_SERVER, null);
  }

  /**
   * Tries to establish a connection to a PRP by using the passed url variable.
   *
   * @param component The Component containing the URL used to connect to the
   *          component, like rmi://localhost:1111/prpRmi
   * @param preferredConnectorType The preferred connector type (optional). If
   *          null, if the preferredConnectorType string is not found or if
   *          there is only one url, the connector will be selected according to
   *          the priorities list. This parameter is not case sensitive.
   * @return A connector suitable for the specified protocol, null in case of an
   *         error
   */
  public static IPolicyRetrievalPoint getPrp(Component component, final String preferredConnectorType) {
    return getConnector(component, ComponentType.PRP, preferredConnectorType, null);
  }

  /**
   * Tries to establish a connection to a PRP by using the passed url variable.
   *
   * @param url The URL used to connect to the component, like
   *          rmi://localhost:1111/prpRmi
   * @return A connector suitable for the specified protocol, null in case of an
   *         error
   */
  public static IPolicyRetrievalPoint getPrp(URI url) {
    return getConnector(url, ComponentType.PRP, null);
  }

  /**
   * Tries to establish a connection to a PXP by using the passed url variable.
   *
   * @param component The Component containing the URL used to connect to the
   *          component, like rmi://localhost:1111/pxpRmi
   * @param preferredConnectorType The preferred connector type (optional). If
   *          null, if the preferredConnectorType string is not found or if
   *          there is only one url, the connector will be selected according to
   *          the priorities list. This parameter is not case sensitive.
   * @return A connector suitable for the specified protocol, null in case of an
   *         error
   */
  public static IPolicyExecutionPoint getPxp(Component component, final String preferredConnectorType) {
    return getConnector(component, ComponentType.PXP, preferredConnectorType, null);
  }

  /**
   * Tries to establish a connection to a PXP by using the passed url variable.
   *
   * @param url The URL used to connect to the component, like
   *          rmi://localhost:1111/pxpRmi
   * @return A connector suitable for the specified protocol, null in case of an
   *         error
   */
  public static IPolicyExecutionPoint getPxp(URI url) {
    return getConnector(url, ComponentType.PXP, null);
  }

  protected static void loadClasses() {
    if (connectorClasses == null) {
      final Reflections reflections = new Reflections("de");
      connectorClasses = reflections.getTypesAnnotatedWith(Connector.class);
    }
  }
}
