/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.policy.parameter;

import de.fraunhofer.iese.ind2uce.api.common.Ind2uceEntity;

import org.apache.commons.lang3.builder.HashCodeBuilder;

import java.lang.reflect.Type;

/**
 * A parameter, basically a key-value pair.
 *
 * @author Fraunhofer IESE
 * @param <T> the type of the value
 */
public class Parameter<T> extends Ind2uceEntity {

  /**
   * The Constant serialVersionUID.
   */
  private static final long serialVersionUID = 6426150182821452997L;

  /**
   * The name of the parameter.
   */
  private String name;

  /**
   * The value of the parameter.
   */
  private DataObject<T> value;

  /**
   * Instantiates a new parameter.
   *
   * @param name the name
   * @param value the value
   */
  public Parameter(String name, T value) {
    if (name == null) {
      throw new IllegalArgumentException("Name required");
    }
    if (value == null) {
      throw new IllegalArgumentException("Value required");
    }
    this.name = name;
    this.value = new DataObject<>(value);
  }

  /**
   * Instantiates a new parameter.
   *
   * @param name the name
   * @param value the value
   * @param clazz the clazz
   */
  public Parameter(String name, T value, Class<T> clazz) {
    if (name == null) {
      throw new IllegalArgumentException("Name required");
    }
    if (value == null) {
      throw new IllegalArgumentException("Value required");
    }
    this.name = name;
    this.value = new DataObject<>(value, clazz);
  }

  /**
   * Instantiates a new parameter.
   *
   * @param name the name
   * @param value the value
   * @param valueType the value type
   */
  public Parameter(String name, T value, Type valueType) {
    if (name == null) {
      throw new IllegalArgumentException("Name required");
    }
    if (value == null) {
      throw new IllegalArgumentException("Value required");
    }
    this.name = name;
    this.value = new DataObject<>(value, valueType);

  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  @SuppressWarnings("rawtypes")
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }

    if (obj == null) {
      return false;
    }

    if (!(obj instanceof Parameter)) {
      return false;
    }

    final Parameter other = (Parameter)obj;

    if (this.name == null) {
      if (other.name != null) {
        return false;
      }
    } else if (!this.name.equals(other.name)) {
      return false;
    }

    if (this.value == null) {
      if (other.value != null) {
        return false;
      }
    } else if (!this.value.equals(other.value)) {
      return false;
    }

    return true;
  }

  // /**
  // * Gets the data object java type.
  // *
  // * @return the data object java type
  // */
  // public Type getDataObjectJavaType() {
  // return this.value.internJavaType();
  // }

  /**
   * Gets the name of the parameter.
   *
   * @return the name of the parameter
   */
  public String getName() {
    return this.name;
  }

  /**
   * Gets the type.
   *
   * @return the type
   */
  public Class<?> getType() {
    return this.value.getType();
  }

  /**
   * Gets the type name.
   *
   * @return the type name
   */
  public String getTypeName() {
    return this.value.getTypeName();
  }

  /**
   * Gets the value of the parameter.
   *
   * @return the value of the parameter
   */
  public T getValue() {
    return this.value.getValue();
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#hashCode()
   */
  @Override
  public int hashCode() {
    final HashCodeBuilder builder = new HashCodeBuilder(17, 31);
    builder.append(this.name);
    if (this.value != null) {
      builder.append(this.value.getValue());
      builder.append(this.value.getType());
    }
    return builder.toHashCode();
  }

  /**
   * Sets the data object java type.
   *
   * @param type the new data object java type
   */
  public void setDataObjectJavaType(Type type) {
    this.value.setJavaType(type);
  }

  /**
   * Sets the name of the parameter.
   *
   * @param name the new name of the parameter
   */
  public void setName(String name) {
    this.name = name;
  }

  /**
   * Sets the value of the parameter.
   *
   * @param value the new value of the parameter
   */
  public void setValue(T value) {
    this.value = new DataObject<>(value);
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.common.Ind2uceEntity#toString()
   */
  @Override
  public String toString() {
    return "  " + this.name + ": " + this.value;
  }

}
