/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.policy.parameter;

import de.fraunhofer.iese.ind2uce.api.common.Ind2uceEntity;
import de.fraunhofer.iese.ind2uce.logger.LoggerFactory;

import com.google.gson.internal.Primitives;

import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.slf4j.Logger;

import java.lang.reflect.Type;

/**
 * A value of a certain {@link DataType} that is currently supported by IND2UCE.
 *
 * @param <T> the type of the value. Must be one of {@link DataType}.
 * @author Fraunhofer IESE
 */
public class DataObject<T> extends Ind2uceEntity {

  /**
   * The Constant serialVersionUID.
   */
  private static final long serialVersionUID = 6426150182821452997L;

  /**
   * The Constant log.
   */
  private static final Logger log = LoggerFactory.getLogger("DataObject<T>");

  /**
   * Indicates a complex type.
   */
  boolean isComplex = false;

  /**
   * The value.
   */
  private T value;

  /**
   * The {@link DataType} of the value.
   */
  private String type;

  /**
   * .
   */
  private transient Type javaType;

  /**
   * Used for JAXB
   */
  public DataObject() {

  }

  /**
   * Instantiates a new data object.
   *
   * @param type the type
   */
  public DataObject(Class<T> type) {
    this.type = type.getCanonicalName();
  }

  /**
   * Instantiates a new data object.
   *
   * @param value the value
   */
  public DataObject(T value) {
    if (value == null) {
      throw new IllegalArgumentException();
    }
    this.value = value;
    if (this.value != null) {
      this.type = value.getClass().getCanonicalName();
    }
  }

  /**
   * Instantiates a new data object.
   *
   * @param value the value
   * @param clazz the clazz
   */
  public DataObject(T value, Class<T> clazz) {
    this(value);
    if (value.getClass() != clazz && !this.primitiveTypesMatches(value, clazz)) {
      throw new IllegalArgumentException("Incompatible data types: " + value.getClass() + ", " + clazz);
    }
    this.type = clazz.getCanonicalName();
  }

  /**
   * *.
   *
   * @param value the value
   * @param type the type
   */
  public DataObject(T value, Type type) {
    this(value);
    this.javaType = type;
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  @SuppressWarnings("rawtypes")
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }

    if (obj == null) {
      return false;
    }
    if (!(obj instanceof DataObject)) {
      return false;
    }

    final DataObject other = (DataObject)obj;

    if (this.type != other.type) {
      return false;
    }

    if (this.value == null) {
      if (other.value != null) {
        return false;
      }
    } else if (!this.value.equals(other.getValue())) {
      return false;
    }

    return true;
  }

  // /**
  // * Gets the java type.
  // *
  // * @return the java type
  // */
  // public Type internJavaType() {
  // return this.javaType;
  // }

  /**
   * Sets the java type.
   *
   * @param javaType the new java type
   */
  public void setJavaType(Type javaType) {
    this.javaType = javaType;
  }

  /**
   * Gets the {@link DataType} of the value.
   *
   * @return the {@link DataType} of the value
   */
  public Class<?> getType() {
    try {
      return Class.forName(this.type);
    } catch (final ClassNotFoundException e) {
      log.error("Class not found for " + this.type, e);
      return null;
    }
  }

  /**
   * Sets the type.
   *
   * @param type the type to set
   */
  public void setType(String type) {
    this.type = type;
  }

  /**
   * Gets the type name.
   *
   * @return the type name
   */
  public String getTypeName() {
    return this.type;
  }

  /**
   * Gets the value.
   *
   * @return the value
   */
  public T getValue() {
    return this.value;
  }

  /**
   * Sets the value.
   *
   * @param value the new value
   */
  public void setValue(T value) {
    this.value = value;
    if (this.value != null) {
      this.type = value.getClass().getCanonicalName();
    }
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#hashCode()
   */
  @Override
  public int hashCode() {
    final HashCodeBuilder builder = new HashCodeBuilder(17, 31);
    builder.append(this.type);
    builder.append(this.value);
    return builder.toHashCode();
  }

  /**
   * Checks if is complex.
   *
   * @return true, if is complex
   */
  public boolean isComplex() {
    return this.isComplex;
  }

  /**
   * Sets the complex.
   *
   * @param b the new complex
   */
  public void setComplex(boolean b) {
    this.isComplex = b;
  }

  /**
   * Primitive types matches.
   *
   * @param value the value
   * @param clazz the clazz
   * @return true, if successful
   */
  private boolean primitiveTypesMatches(T value, Class<T> clazz) {
    return Primitives.isWrapperType(value.getClass()) && Primitives.unwrap(value.getClass()) == clazz || Primitives.isWrapperType(clazz) && Primitives.unwrap(clazz) == value.getClass();
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.common.Ind2uceEntity#toString()
   */
  @Override
  public String toString() {
    return this.value == null ? null : this.value.toString();
  }

}
