/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.policy.identifier;

import de.fraunhofer.iese.ind2uce.api.policy.PolicyType;

import java.util.Arrays;
import java.util.Iterator;
import java.util.regex.Pattern;

/**
 * The Class PolicyIdUtil.
 *
 * @author Bernd Peltzer-Bartels
 */
public class PolicyIdUtil {

  /**
   * Pattern that matches IND²UCE policies
   * urn:policy:&lt;scope&gt;:&lt;name&gt;.
   */
  private static final Pattern URN_POLICY_PATTERN = Pattern.compile("^urn:policy(:([A-Za-z0-9()+,\\-.=@;$_!*']+)){2}$", Pattern.CASE_INSENSITIVE);

  /**
   * Creates the policy id.
   *
   * @param urn the urn
   * @return the policy id
   */
  public static PolicyId createPolicyId(String urn) {
    final PolicyId policyId = new PolicyId();
    setAttributesByUrn(urn, policyId);
    return policyId;
  }

  /**
   * Sets the attributes.
   *
   * @param urn the urn
   * @param policyId the policy id
   */
  private static void setAttributes(String urn, PolicyId policyId) {
    final String[] result = urn.split(":");
    if (result.length > 2) {
      policyId.setScope(result[2]);
    }
    if (result.length > 3) {
      final Iterator<String> iter = Arrays.asList(Arrays.copyOfRange(result, 3, result.length)).iterator();
      final StringBuilder s = new StringBuilder();
      while (iter.hasNext()) {
        s.append(iter.next());
        if (iter.hasNext()) {
          s.append(":");
        }
      }
      policyId.setIdentifier(s.toString());
    }
  }

  /**
   * To set attribute by urn.
   *
   * @param urn the urn
   * @param instance the instance
   */
  public static void setAttributesByUrn(String urn, PolicyId instance) {
    if (URN_POLICY_PATTERN.matcher(urn).matches()) {
      setILPAttributesByUrn(urn, instance);
    } else {
      throw new IllegalArgumentException("URN " + urn + " does not match the pattern " + URN_POLICY_PATTERN);
    }
  }

  /**
   * Sets the ILP attributes by urn.
   *
   * @param urn the urn
   * @param policyId the policy id
   * @return the policy id
   */
  private static PolicyId setILPAttributesByUrn(String urn, PolicyId policyId) {
    setAttributes(urn, policyId);
    policyId.setPolicyType(PolicyType.ILP);
    return policyId;
  }

}
