/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.policy.identifier;

import de.fraunhofer.iese.ind2uce.api.common.Ind2uceEntity;
import de.fraunhofer.iese.ind2uce.api.component.identifier.EnforcementScopeId;
import de.fraunhofer.iese.ind2uce.api.policy.PolicyType;

import javax.persistence.Column;
import javax.persistence.Embeddable;
import javax.persistence.EnumType;
import javax.persistence.Enumerated;

/**
 * The Class PolicyId.
 *
 * @author Bernd Peltzer-Bartels
 */
@Embeddable
public class PolicyId extends Ind2uceEntity {

  /**
   * The Constant serialVersionUID.
   */
  private static final long serialVersionUID = 4279735491466729161L;

  /**
   * The type of the policy the ID belongs to.
   */

  @Enumerated(EnumType.STRING)
  @Column(name = "policy_type", length = 8)
  private PolicyType policyType = null;

  /**
   * The scope of the policy.
   */
  @Column(name = "scope", length = 70)
  private String scope;

  /**
   * The identifier.
   */
  @Column(name = "identifier", length = 70)
  private String identifier;

  /**
   * Instantiates a new policy id.
   */
  public PolicyId() {

  }

  /**
   * Instantiates a new policy id.
   *
   * @param urn the urn
   */
  public PolicyId(String urn) {
    PolicyIdUtil.setAttributesByUrn(urn, this);
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (!(o instanceof PolicyId)) {
      return false;
    }

    final PolicyId policyId = (PolicyId)o;

    if (this.policyType != policyId.policyType) {
      return false;
    }
    if (this.scope != null ? !this.scope.equals(policyId.scope) : policyId.scope != null) {
      return false;
    }
    return this.identifier != null ? this.identifier.equals(policyId.identifier) : policyId.identifier == null;

  }

  /**
   * Gets the es type.
   *
   * @return the es type
   */
  public EnforcementScopeId getEsType() {
    if (this.policyType == PolicyType.ILP) {
      return new EnforcementScopeId("urn:es:" + this.scope);
      // return PolicyIdUtil.createPolicyId("urn:slp:" + platform);
    }
    return null;
  }

  /**
   * Gets the id.
   *
   * @return the id
   */
  public String getId() {
    return this.toString();
  }

  /**
   * Gets the identifier.
   *
   * @return the identifier
   */
  public String getIdentifier() {
    return this.identifier;
  }

  /**
   * Gets the type of the policy the ID belongs to.
   *
   * @return the policyType
   */
  public PolicyType getPolicyType() {
    return this.policyType;
  }

  /**
   * Gets the platform.
   *
   * @return the platform
   */
  public String getScope() {
    return this.scope;
  }

  /**
   * Gets the type.
   *
   * @return the type
   */
  public PolicyType getType() {
    return this.policyType;
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#hashCode()
   */
  @Override
  public int hashCode() {
    int result = this.policyType != null ? this.policyType.hashCode() : 0;
    result = 31 * result + (this.scope != null ? this.scope.hashCode() : 0);
    result = 31 * result + (this.identifier != null ? this.identifier.hashCode() : 0);
    return result;
  }

  /**
   * Sets the id.
   *
   * @param uri the new id
   */
  public void setId(String uri) {
    PolicyIdUtil.setAttributesByUrn(uri, this);
  }

  /**
   * Sets the identifier.
   *
   * @param identifier the new identifier
   */
  public void setIdentifier(String identifier) {
    this.identifier = identifier;
  }

  /**
   * Sets the type of the policy the ID belongs to.
   *
   * @param policyType the new type of the policy the ID belongs to
   */
  public void setPolicyType(PolicyType policyType) {
    this.policyType = policyType;
  }

  /**
   * Sets the platform.
   *
   * @param scope the new platform
   */
  public void setScope(String scope) {
    this.scope = scope;
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.common.Ind2uceEntity#toString()
   */
  @Override
  public String toString() {
    if (this.policyType == null || this.identifier == null || this.scope == null) {
      return "PolicyId not initialized.";
    }
    final StringBuilder s = new StringBuilder();
    s.append("urn:");
    if (this.getType() == PolicyType.ILP) {
      s.append("policy");
    } else {
      s.append(this.getType().toString().toLowerCase());
    }
    s.append(":");
    s.append(this.getScope());
    s.append(":");
    s.append(this.getIdentifier());
    return s.toString();
  }
}
