/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.policy.identifier;

import de.fraunhofer.iese.ind2uce.api.common.Ind2uceEntity;
import de.fraunhofer.iese.ind2uce.api.policy.Event;

import java.util.regex.Pattern;

import javax.persistence.Column;
import javax.persistence.Embeddable;

/**
 * The ActionId uniquely identifies an action that can occur in a certain system
 * on a certain abstraction level. Examples might be: *
 * <ul>
 * <li>urn:action:laptop123:filesystem-open</li>
 * <li>urn:action:chrome231:ui-click</li>
 * <li>urn:action:android456:os-broadcast</li>
 * </ul>
 * The format must adhere to the {@link ActionId#URN_ACTION_PATTERN}. Concrete
 * actions that occurred are represented by an {@link Event}.
 */
@Embeddable
public class ActionId extends Ind2uceEntity {

  /**
   * The Constant serialVersionUID.
   */
  private static final long serialVersionUID = 4279735491466729161L;

  /**
   * The URN pattern for an ActionId. Must be of the format
   * &quot;urn:action:&lt;scope&gt;:&lt;id&gt;&quot; where
   * <ul>
   * <li>scope is the scope (e.g., system, domain) where the action occurs</li>
   * <li>id is the unique id of the action</li>
   * </ul>
   */
  public static final Pattern URN_ACTION_PATTERN = Pattern.compile("^urn:action(:[A-Za-z0-9()+,\\-.=@;$_!*']+){2}$", Pattern.CASE_INSENSITIVE);

  /**
   * The platform.
   */
  @Column(name = "action_scope")
  private String scope;

  /**
   * The action.
   */
  @Column(name = "action")
  private String action;

  /**
   * Instantiates a new action id.
   */
  ActionId() {
    super();
  }

  /**
   * Instantiates a new action id.
   *
   * @param urn the urn. Must adhere to the {@link #URN_ACTION_PATTERN}.
   */
  public ActionId(String urn) {
    this.validate(urn);
    final String[] s = urn.split(":");
    assert (s.length > 3);
    assert (s[0].equalsIgnoreCase("urn"));
    assert (s[1].equalsIgnoreCase("action"));
    this.scope = s[2];
    this.action = s[3];
  }

  /**
   * Instantiates a new action id.
   *
   * @param scope the scope of the action
   * @param actionId the unique action id
   */
  public ActionId(String scope, String actionId) {
    this.scope = scope;
    this.action = actionId;
    this.validate(this.toString());
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (!(o instanceof ActionId)) {
      return false;
    }

    final ActionId actionId = (ActionId)o;

    if (this.scope != null ? !this.scope.equals(actionId.scope) : actionId.scope != null) {
      return false;
    }
    return this.action != null ? this.action.equals(actionId.action) : actionId.action == null;

  }

  /**
   * UI, OS, FILESYSTEM etc.
   *
   * @return get action name
   */
  public String getAction() {
    return this.action;
  }

  /**
   * Gets the id.
   *
   * @return the id
   */
  public String getId() {
    return this.toString();
  }

  /**
   * ANDROID, OSX etc.
   *
   * @return platform details
   */
  public String getScope() {
    return this.scope;
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#hashCode()
   */
  @Override
  public int hashCode() {
    int result = this.scope != null ? this.scope.hashCode() : 0;
    result = 31 * result + (this.action != null ? this.action.hashCode() : 0);
    return result;
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.common.Ind2uceEntity#toString()
   */
  @Override
  public String toString() {
    final StringBuilder s = new StringBuilder();
    s.append("urn:action:");
    s.append(this.scope);
    s.append(":");
    s.append(this.action);
    return s.toString();
  }

  /**
   * Validate.
   *
   * @param uri the uri
   */
  /*
   * (non-Javadoc)
   * @see
   * de.fraunhofer.iese.ind2uce.api.common.UniqueIdentifier#validate(java.net.
   * URI)
   */
  protected void validate(String uri) {
    URN_ACTION_PATTERN.matcher(uri).matches();
  }
}
