/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.policy;

import de.fraunhofer.iese.ind2uce.api.component.identifier.EnforcementScopeId;
import de.fraunhofer.iese.ind2uce.logger.LoggerFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.ErrorHandler;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

import java.io.IOException;
import java.io.StringReader;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import javax.xml.XMLConstants;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

/**
 * The Class TimerValidator.
 */
class TimerValidator46 implements ITimerValidator {

  /**
   * The Constant LOG.
   */
  private static final org.slf4j.Logger LOG = LoggerFactory.getLogger(TimerValidator46.class);

  /**
   * The Constant SCHEMA_RESOURCE_FILEPATH.
   */
  private static final String SCHEMA_RESOURCE_FILEPATH = "/languageSchema3_2/ind2uceLanguageTimer.xsd";

  /**
   * The schema.
   */
  private Schema schema;

  /**
   * The validator.
   */
  private Validator validator;

  public TimerValidator46() {
    try {
      final SchemaFactory schemaFactory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
      final URL url = TimerValidator46.class.getResource(SCHEMA_RESOURCE_FILEPATH);
      this.schema = schemaFactory.newSchema(url);

      this.validator = this.schema.newValidator();

      this.validator.setErrorHandler(new ErrorHandler() {
        @Override
        public void error(SAXParseException exception) throws SAXException {
          LOG.error("Validation error: " + exception.getMessage());
          throw exception;
        }

        @Override
        public void fatalError(SAXParseException exception) throws SAXException {
          LOG.error("Validation fatal error: " + exception.getMessage());
          throw exception;
        }

        @Override
        public void warning(SAXParseException exception) throws SAXException {
          LOG.error("Validation warning: " + exception.getMessage());
          throw exception;
        }
      });

      LOG.info("Successfully loaded schema");
    } catch (final SAXException e) {
      LOG.error("Unable to create schema", e);
    }
  }

  @Override
  public boolean validateXMLSchema(String timerXML) throws InvalidTimerException {

    try {

      this.validator.validate(new StreamSource(new StringReader(timerXML)));
    } catch (IOException | SAXException e) {
      LOG.info("Exception: " + e.getMessage());
      throw new InvalidTimerException("Timer is not valid according to XML Schema", e);
    }
    return true;
  }

  @Override
  public boolean checkTimerSolution(Timer t) throws InvalidTimerException {

    final String solutionId = t.getScope();

    try {
      final DocumentBuilderFactory documentumentBuilderFactory = DocumentBuilderFactory.newInstance();
      documentumentBuilderFactory.setNamespaceAware(true);
      final DocumentBuilder documentumentBuilder = documentumentBuilderFactory.newDocumentBuilder();
      final Document document = documentumentBuilder.parse(new InputSource(new StringReader(t.getXml())));
      final XPathFactory xpathFactory = XPathFactory.newInstance();
      final XPath xpath = xpathFactory.newXPath();

      final HashMap<String, String> prefMap = new HashMap<String, String>() {
        {
          this.put("tns", "http://www.iese.fraunhofer.de/ind2uce/3.2.46/ind2uceLanguageTimer");
          this.put("pip", "http://www.iese.fraunhofer.de/ind2uce/3.2.46/pip");
          this.put("parameter", "http://www.iese.fraunhofer.de/ind2uce/3.2.46/parameter");
          this.put("event", "http://www.iese.fraunhofer.de/ind2uce/3.2.46/event");
          this.put("constant", "http://www.iese.fraunhofer.de/ind2uce/3.2.46/constant");
        }
      };
      final SimpleNamespaceContext namespaces = new SimpleNamespaceContext(prefMap);
      xpath.setNamespaceContext(namespaces);

      // check PEP events
      XPathExpression expr = xpath.compile(".//tns:mechanism");
      NodeList names = (NodeList)expr.evaluate(document, XPathConstants.NODESET);

      final List<String> errors = new ArrayList<>();

      for (int i = 0; i < names.getLength(); i++) {
        final Node n = names.item(i);
        final String actionName = n.getAttributes().getNamedItem("event").getNodeValue();
        if (actionName.split(":").length < 3) {
          errors.add("Event " + actionName + " does not seem to contain enough : .");
        } else {
          final String v = actionName.split(":")[2];
          if (!new EnforcementScopeId(v).getIdentifier().equals(solutionId)) {
            errors.add("Event " + actionName + " does not refer to an action of solution " + solutionId + ".");
          }
        }
      }

      // check PIP methods
      expr = xpath.compile(".//pip:string|.//pip:boolean|.//pip:number|.//pip:object|.//pip:list");
      names = (NodeList)expr.evaluate(document, XPathConstants.NODESET);
      for (int i = 0; i < names.getLength(); i++) {
        final Node n = names.item(i);
        final String actionName = n.getAttributes().getNamedItem("method").getNodeValue();
        if (actionName.split(":").length < 3) {
          errors.add("PIP " + actionName + " does not seem to contain enough : .");
        } else {
          final String v = actionName.split(":")[2];
          if (!new EnforcementScopeId(v).getIdentifier().equals(solutionId)) {
            errors.add("PIP " + actionName + " does not belong to solution " + solutionId + ".");
          }
        }
      }

      // check PXP methods
      expr = xpath.compile(".//tns:execute");
      names = (NodeList)expr.evaluate(document, XPathConstants.NODESET);
      for (int i = 0; i < names.getLength(); i++) {
        final Node n = names.item(i);
        final String methodName = n.getAttributes().getNamedItem("action").getNodeValue();
        final String v = methodName.split(":")[2];
        if (!new EnforcementScopeId(v).getIdentifier().equals(solutionId)) {
          errors.add("ExecuteAction " + methodName + " does not refer to a PXP of solution " + solutionId + ".");
        }
      }

      if (errors.size() > 0) {
        final StringBuilder b = new StringBuilder();
        b.append("Timer is invalid due to the following errors:\n");
        for (final String error : errors) {
          b.append(error);
          b.append("\n");
        }
        throw new InvalidTimerException(b.toString());
      }

    } catch (IOException | SAXException | XPathExpressionException | ParserConfigurationException e) {
      throw new InvalidTimerException("Timer validation failed", e);
    }

    return true;
  }

}
