/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2017 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.policy;

import de.fraunhofer.iese.ind2uce.api.policy.identifier.ActionId;
import de.fraunhofer.iese.ind2uce.api.policy.parameter.Parameter;
import de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterList;
import de.fraunhofer.iese.ind2uce.logger.LoggerFactory;

import org.apache.xerces.dom.DeferredElementImpl;
import org.slf4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.regex.Pattern;

import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.Id;
import javax.persistence.Lob;
import javax.persistence.Transient;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

@Entity
public class Timer {

  /**
   * The logger.
   */
  private static final Logger LOG = LoggerFactory.getLogger(Timer.class);

  private static final String VALUE = "value";

  public static final Pattern URN_TIMER_PATTERN = Pattern.compile("^urn:timer(:[A-Za-z0-9()+,\\-.=@;$_!*']+){2}$", Pattern.CASE_INSENSITIVE);

  @Column(name = "identifier", length = 70)
  private String identifier;

  /**
   * Cron value for this
   */
  @Column(length = 64, nullable = false)
  private String cronValue;

  /**
   * The deployed.
   */
  @Column(nullable = false)
  private boolean deployed;

  /**
   * The description.
   */
  @Column(length = 512)
  private String description;

  /**
   * Timer policy as XML.
   */
  @Lob
  private String xml;

  /**
   * Scope of timer.
   */
  @Column(length = 255)
  private String scope;

  /**
   * URN id of timer.
   */
  @Id
  @Column(length = 255)
  private String id;

  /**
   * List of events to be executed when timer is executed.
   */
  @Transient
  private List<Event> events;

  @Transient
  private boolean xmlValid = false;

  @Transient
  private boolean scopeValid = false;

  @Transient
  private InvalidTimerException invalidTimerException;

  /**
   * @return the isXmlValid
   */
  public boolean isXmlValid() {
    return xmlValid;
  }

  public InvalidTimerException getInvalidTimerException() {
    return this.invalidTimerException;
  }

  /**
   * @param isXmlValid the isXmlValid to set
   */
  public void setXmlValid(boolean xmlValid) {
    this.xmlValid = xmlValid;
  }

  /**
   * @param isScopeValid the isScopeValid to set
   */
  public void setScopeValid(boolean isScopeValid) {
    this.scopeValid = isScopeValid;
  }

  /**
   * @param invalidTimerException the invalidTimerException to set
   */
  public void setInvalidTimerException(InvalidTimerException invalidTimerException) {
    this.invalidTimerException = invalidTimerException;
  }

  public boolean isScopeValid() {
    return this.scopeValid;
  }

  public Timer(String xml) throws InvalidTimerException {
    super();
    this.xml = xml;
    this.validateXML();
    this.parse();
    this.validateScope();
  }

  public void buildFromURN(String urn) {
    this.validate(urn);
    final String[] s = urn.split(":");
    assert (s.length > 3);
    assert (s[0].equalsIgnoreCase("urn"));
    assert (s[1].equalsIgnoreCase("timer"));
    this.scope = s[2];
    this.identifier = s[3];
    id = urn;
  }

  protected void validate(String uri) {
    if (!URN_TIMER_PATTERN.matcher(uri).matches()) {
      throw new IllegalArgumentException("TimerId " + uri + " do not match URN: " + URN_TIMER_PATTERN);
    }
  }

  private void parse() throws InvalidTimerException {
    try {
      final DocumentBuilderFactory documentumentBuilderFactory = DocumentBuilderFactory.newInstance();
      documentumentBuilderFactory.setNamespaceAware(false);
      final DocumentBuilder documentumentBuilder = documentumentBuilderFactory.newDocumentBuilder();
      final Document document = documentumentBuilder.parse(new InputSource(new StringReader(xml)));
      final XPathFactory xpathFactory = XPathFactory.newInstance();
      final XPath xpath = xpathFactory.newXPath();

      // Fetch metadata and initialize timer.
      buildFromURN((String)xpath.compile("//timer/@id").evaluate(document, XPathConstants.STRING));
      this.setCronValue((String)xpath.compile("//timer/@cron").evaluate(document, XPathConstants.STRING));
      this.setDescription((String)xpath.compile("//timer/@description").evaluate(document, XPathConstants.STRING));

      // Fetch event data
      final XPathExpression expr = xpath.compile("//timer/event");
      final NodeList names = (NodeList)expr.evaluate(document, XPathConstants.NODESET);

      events = new ArrayList<>();

      for (int i = 0; i < names.getLength(); i++) {
        events.add(this.parseEvent(names.item(i)));
      }
    } catch (IOException | ParserConfigurationException | SAXException | XPathExpressionException e) {
      throw new InvalidTimerException("The Timer was not valid");
    }

  }

  private Event parseEvent(Node item) {
    final String action = item.getAttributes().getNamedItem("action").getNodeValue();
    final ParameterList params = new ParameterList();

    for (final Parameter p : this.getParametersByTagName(item, "string")) {
      params.add(p);
    }

    for (final Parameter p : this.getParametersByTagName(item, "boolean")) {
      params.add(p);
    }

    for (final Parameter p : this.getParametersByTagName(item, "number")) {
      params.add(p);
    }

    for (final Parameter p : this.getParametersByTagName(item, "object")) {
      params.add(p);
    }

    for (final Parameter p : this.getParametersByTagName(item, "list")) {
      params.add(p);
    }

    return new Event(new ActionId(action), true, System.currentTimeMillis(), params);
  }

  private Collection<Parameter> getParametersByTagName(Node item, String string) {
    final Collection<Parameter> params = new ArrayList<>();

    final NodeList elementsByTagName = ((DeferredElementImpl)item).getElementsByTagName(string);

    for (int i = 0; i < elementsByTagName.getLength(); i++) {
      final Node node = elementsByTagName.item(i);
      final String name = node.getAttributes().getNamedItem("name").getNodeValue();
      Object value = null;
      switch (string) {
        case "string":
          value = node.getAttributes().getNamedItem(VALUE).getNodeValue();
          break;
        case "number":
          value = Double.valueOf(node.getAttributes().getNamedItem(VALUE).getNodeValue());
          break;
        case "boolean":
          value = Boolean.valueOf(node.getAttributes().getNamedItem(VALUE).getNodeValue());
          break;
        default:
          LOG.info("No match found for string [ method: getParametersByTagName | class TimerService]");
      }
      if (null != value) {
        params.add(new Parameter(name, value));
      }
    }
    return params;
  }

  public Timer() {
    // TODO Auto-generated constructor stub
  }

  public String getCronValue() {
    return this.cronValue;
  }

  public void setCronValue(String cronValue) {
    this.cronValue = cronValue;
  }

  public boolean isDeployed() {
    return this.deployed;
  }

  public void setDeployed(boolean deployed) {
    this.deployed = deployed;
  }

  public String getDescription() {
    return this.description;
  }

  public void setDescription(String description) {
    this.description = description;
  }

  public String getXml() {
    return this.xml;
  }

  public void setXml(String xml) {
    this.xml = xml;
  }

  public List<Event> getEvents() throws InvalidTimerException {
    if (this.events == null || this.events.isEmpty()) {
      parse();
    }
    return this.events;
  }

  public void setEvents(List<Event> events) {
    this.events = events;
  }

  public String getScope() {
    return this.scope;
  }

  public void setScope(String scope) {
    this.scope = scope;
  }

  public String getIdentifier() {
    return this.identifier;
  }

  public void setIdentifier(String identifier) {
    this.identifier = identifier;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || this.getClass() != o.getClass()) {
      return false;
    }

    final Timer timer = (Timer)o;

    return this.identifier.equals(timer.identifier);
  }

  private boolean validateScope() {
    try {
      this.scopeValid = new TimerValidator().checkTimerSolution(this);
    } catch (final InvalidTimerException e) {
      this.scopeValid = false;
      this.invalidTimerException = e;
    } catch (final Exception e) {
      LOG.warn("Exception in method validateScope", e);
    }
    return this.scopeValid;

  }

  public void extractIdAndDescription() {
    try {
      buildFromURN(this.readAttribute("//timer/@id"));
      this.description = this.readAttribute("//timer/@description");
    } catch (final Exception e) {
      LOG.error("Exception in method extractIdAndDescription", e);
    }
  }

  /**
   * Read attribute.
   *
   * @param xpathString the xpath string
   * @return the string
   */
  private String readAttribute(String xpathString) {
    final DocumentBuilderFactory documentumentBuilderFactory = DocumentBuilderFactory.newInstance();
    documentumentBuilderFactory.setNamespaceAware(false);
    try {
      final DocumentBuilder documentumentBuilder = documentumentBuilderFactory.newDocumentBuilder();
      final Document document = documentumentBuilder.parse(new InputSource(new StringReader(this.xml)));
      final XPathFactory xpathFactory = XPathFactory.newInstance();
      final XPath xpath = xpathFactory.newXPath();
      final XPathExpression expr = xpath.compile(xpathString);

      return (String)expr.evaluate(document, XPathConstants.STRING);
    } catch (SAXException | IOException | ParserConfigurationException | XPathExpressionException e) {
      throw new IllegalArgumentException("Timer does not have attribute", e);
    }
  }

  public boolean validateXML() {
    try {
      this.xmlValid = new TimerValidator().validateXMLSchema(this.xml);
    } catch (final InvalidTimerException e) {
      this.xmlValid = false;
      this.invalidTimerException = e;
    }
    return this.xmlValid;
  }

  @Override
  public String toString() {
    return id;
  }

  /**
   * @return the id
   */
  public String getId() {
    return id;
  }

  public void setId(String string) {
    this.id = string;

  }
}
