/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.policy;

import de.fraunhofer.iese.ind2uce.api.policy.identifier.PolicyId;
import de.fraunhofer.iese.ind2uce.logger.LoggerFactory;

import org.slf4j.Logger;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import java.io.IOException;
import java.io.StringReader;
import java.util.HashMap;
import java.util.Map;

import javax.persistence.CollectionTable;
import javax.persistence.ElementCollection;
import javax.persistence.EmbeddedId;
import javax.persistence.Entity;
import javax.persistence.JoinColumn;
import javax.persistence.Lob;
import javax.persistence.Transient;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

/**
 * The Class Policy.
 */
@Entity
public class Policy {
  /**
   * The logger.
   */
  private static final Logger LOG = LoggerFactory.getLogger(Policy.class);

  /**
   * The id.
   */
  @EmbeddedId
  private PolicyId id;

  @Transient
  private boolean xmlValid = false;

  @Transient
  private boolean scopeValid = false;

  public boolean isScopeValid() {
    return this.scopeValid;
  }

  @Transient
  private InvalidPolicyException invalidPolicyException;

  /**
   * The deployed.
   */
  private boolean deployed;

  public InvalidPolicyException getInvalidPolicyException() {
    return this.invalidPolicyException;
  }

  /**
   * The policy description.
   */
  private String description;

  /**
   * The meta data.
   */
  @ElementCollection
  @CollectionTable(name = "policy_meta_data", joinColumns = {
      @JoinColumn(referencedColumnName = "policy_type", name = "policy_type"), @JoinColumn(referencedColumnName = "scope", name = "scope"),
      @JoinColumn(referencedColumnName = "identifier", name = "identifier"),
  })
  private Map<String, String> metaData = new HashMap<>();

  /**
   * The policy.
   */
  @Lob
  private String policy;

  /**
   * Required for JPA.
   */
  public Policy() {

  }

  /**
   * Instantiates a new policy.
   *
   * @param id the id
   * @param policy the policy
   * @deprecated
   */
  @Deprecated
  public Policy(PolicyId id, String policy) {
    super();
    this.id = id;
    this.policy = policy;
    this.validateXML();

    this.validateScope();

  }

  /**
   * Instantiates a new policy.
   *
   * @param id the id
   * @param policy the policy
   * @param description the description
   * @deprecated
   */
  @Deprecated
  public Policy(PolicyId id, String policy, String description) {
    super();
    this.id = id;
    this.policy = policy;
    this.description = description;
    this.validateXML();

    this.validateScope();

  }

  /**
   * Instantiates a new policy.
   *
   * @param policy the policy
   * @throws InvalidPolicyException
   */
  public Policy(String policy) {
    super();
    this.policy = policy;
    this.validateXML();
    this.extractIdAndDescription();

    this.validateScope();

  }

  public boolean isXmlValid() {
    return this.xmlValid;
  }

  private boolean validateScope() {
    try {
      this.scopeValid = new PolicyValidator().checkPolicySolution(this);
    } catch (final InvalidPolicyException e) {
      this.scopeValid = false;
      this.invalidPolicyException = e;
    } catch (final Exception e) {
      LOG.warn("Exception in method validateScope", e);
    }
    return this.scopeValid;

  }

  public void extractIdAndDescription() {
    try {
      this.id = new PolicyId(this.readAttribute("//policy/@id"));
      this.description = this.readAttribute("//policy/@description");
    } catch (final Exception e) {
      LOG.error("Exception in method extractIdAndDescription", e);
    }
  }

  public boolean validateXML() {
    try {
      this.xmlValid = new PolicyValidator().validateXMLSchema(this.policy);
    } catch (final InvalidPolicyException e) {
      this.xmlValid = false;
      this.invalidPolicyException = e;
    }
    return this.xmlValid;
  }

  /**
   * Instantiates a new policy.
   *
   * @param policy the policy
   * @param description the description
   * @throws InvalidPolicyException
   */
  public Policy(String policy, String description) {
    this.policy = policy;
    this.description = description;
    this.validateXML();

    this.extractIdAndDescription();
    this.validateScope();
  }

  /**
   * Adds the meta.
   *
   * @param key the key
   * @param value the value
   */
  public void addMeta(String key, String value) {
    if (this.metaData == null) {
      this.metaData = new HashMap<>();
    }
    this.metaData.put(key, value);
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (!(o instanceof Policy)) {
      return false;
    }

    final Policy thePolicy = (Policy)o;

    return this.id != null ? this.id.equals(thePolicy.id) : thePolicy.id == null;

  }

  /**
   * Gets the description.
   *
   * @return the description
   */
  public String getDescription() {
    return this.description;
  }

  /**
   * Gets the id.
   *
   * @return the id
   */
  public PolicyId getId() {
    return this.id;
  }

  /**
   * Gets the meta data.
   *
   * @return the meta data
   */
  public Map<String, String> getMetaData() {
    return this.metaData;
  }

  /**
   * Gets the policy.
   *
   * @return the policy
   */
  public String getPolicy() {
    return this.policy;
  }

  /**
   * Gets the type.
   *
   * @return the type
   */
  public PolicyType getType() {
    return this.id.getType();
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#hashCode()
   */
  @Override
  public int hashCode() {
    return this.id != null ? this.id.hashCode() : 0;
  }

  /**
   * Checks if is deployed.
   *
   * @return the status
   */
  public boolean isDeployed() {
    return this.deployed;
  }

  /**
   * Read attribute.
   *
   * @param xpathString the xpath string
   * @return the string
   */
  private String readAttribute(String xpathString) {
    try {
      final DocumentBuilderFactory documentumentBuilderFactory = DocumentBuilderFactory.newInstance();
      documentumentBuilderFactory.setNamespaceAware(false);
      final DocumentBuilder documentumentBuilder = documentumentBuilderFactory.newDocumentBuilder();
      final Document document = documentumentBuilder.parse(new InputSource(new StringReader(this.policy)));
      final XPathFactory xpathFactory = XPathFactory.newInstance();
      final XPath xpath = xpathFactory.newXPath();
      final XPathExpression expr = xpath.compile(xpathString);

      return (String)expr.evaluate(document, XPathConstants.STRING);

    } catch (IOException | SAXException | XPathExpressionException | ParserConfigurationException e) {
      throw new IllegalArgumentException("Policy does not have attribute", e);
    }
  }

  /**
   * Sets the deployed.
   *
   * @param status the status to set
   */
  public void setDeployed(boolean status) {
    this.deployed = status;
  }

  /**
   * Sets the description.
   *
   * @param description the description
   */
  public void setDescription(String description) {
    this.description = description;
  }

  /**
   * Sets the id.
   *
   * @param id the new id
   */
  public void setId(PolicyId id) {
    this.id = id;
  }

  /**
   * Sets the meta data.
   *
   * @param metaData the meta data
   */
  public void setMetaData(Map<String, String> metaData) {
    this.metaData = metaData;
  }

  /**
   * Sets the policy.
   *
   * @param policy the new policy
   */
  public void setPolicy(String policy) {
    this.policy = policy;
  }
}
