/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.policy;

import de.fraunhofer.iese.ind2uce.api.common.Ind2uceEntity;
import de.fraunhofer.iese.ind2uce.api.policy.parameter.DataObject;
import de.fraunhofer.iese.ind2uce.api.policy.parameter.Parameter;
import de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterList;
import de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider;
import de.fraunhofer.iese.ind2uce.logger.LoggerFactory;

import org.slf4j.Logger;

/**
 * A request that instructs the PIP.
 *
 * @author Fraunhofer IESE
 * @param <T> the expected return type of the evaluation
 */
public class PipRequest<T> extends Ind2uceEntity implements ParameterListProvider {

  /**
   * The Constant serialVersionUID.
   */
  private static final long serialVersionUID = 938365002647331506L;

  /**
   * The Constant LOG.
   */
  private static final Logger LOG = LoggerFactory.getLogger(PipRequest.class);

  /**
   * The name of the method to be evaluated.
   */
  private String name = null;

  /**
   * The default value that is returned in case of errors.
   */
  private DataObject<T> defaultValue;

  /**
   * A set of parameters that are used for evaluation.
   */
  private ParameterList parameters = new ParameterList();

  /**
   * Instantiates a new pip request.
   *
   * @param name the name of the method to be executed
   * @param defaultValue the default value for error cases
   * @param params parameters used for evaluation
   */
  public PipRequest(String name, T defaultValue, Parameter<?>... params) {
    this(name, defaultValue, new ParameterList(params));
  }

  /**
   * Instantiates a new pip request.
   *
   * @param name the name of the method to be executed
   * @param defaultValue the default value for error cases
   * @param params parameters used for evaluation
   */
  public PipRequest(String name, T defaultValue, ParameterList params) {
    this.name = name;
    this.defaultValue = new DataObject<>(defaultValue);
    if (params != null) {
      this.parameters = params;
    }
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * addParameter(de.fraunhofer.iese.ind2uce.api.policy.parameter.Parameter)
   */
  @Override
  public void addParameter(Parameter<?> param) {
    this.parameters.add(param);
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * addParameter(java.lang.String, java.lang.Object)
   */
  @Override
  public <R> void addParameter(String name, R value) {
    this.parameters.add(new Parameter<>(name, value));
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * clearParameters()
   */
  @Override
  public void clearParameters() {
    this.parameters.clear();
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  public boolean equals(Object obj) {
    if (obj == null) {
      return false;
    }
    if (!(obj instanceof PipRequest)) {
      return false;
    }

    @SuppressWarnings("rawtypes")
    final PipRequest<?> pip2 = (PipRequest)obj;
    LOG.debug("Comparing pipRequest: " + this.getName());

    if (!this.name.equals(pip2.getName())) {
      return false;
    }

    LOG.debug("Comparing parameter: this:" + this.parameters.size() + "; other: " + pip2.parameters.size());
    if (this.parameters.size() != pip2.parameters.size()) {
      return false;
    }

    for (final Parameter<?> curParam : this.parameters) {
      try {
        if (!curParam.equals(pip2.getParameterForName(curParam.getName()))) {
          LOG.debug("param " + curParam.getName() + " differ: [" + curParam.getValue() + "] vs. [" + pip2.getParameterForName(curParam.getName()).getValue() + "]");
          return false;
        }
      } catch (final NullPointerException e) {
        final String msg = "Param: " + curParam.getName() + " is NOT present.";
        LOG.debug(msg);
        LOG.trace(msg, e);
        return false;
      }
    }

    return true;
  }

  /**
   * Gets the default value that is returned in case of errors.
   *
   * @return the default value that is returned in case of errors
   */
  public DataObject<?> getDefaultValue() {
    return this.defaultValue;
  }

  /**
   * Gets the name of the method to be evaluated.
   *
   * @return the name of the method to be evaluated
   */
  public String getName() {
    return this.name;
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * getParameterForName(java.lang.String)
   */
  @Override
  public Parameter<?> getParameterForName(String name) {
    return this.parameters.getParameterForName(name);
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * getParameters()
   */
  @Override
  public ParameterList getParameters() {
    return this.parameters;
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * getParameterValue(java.lang.String, java.lang.Class)
   */
  @Override
  public <R> R getParameterValue(String name, Class<R> clazz) {
    return this.parameters.getParameterValue(name, clazz);
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * removeParameter(java.lang.String)
   */
  @Override
  public void removeParameter(String name) {
    this.parameters.remove(name);
  }

  /**
   * Sets the default value that is returned in case of errors.
   *
   * @param defaultValue the new default value that is returned in case of
   *          errors
   */
  public void setDefaultValue(DataObject<T> defaultValue) {
    this.defaultValue = defaultValue;
  }

  /**
   * Sets the name of the method to be evaluated.
   *
   * @param name the new name of the method to be evaluated
   */
  public void setName(String name) {
    this.name = name;
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * setParameters(de.fraunhofer.iese.ind2uce.api.policy.parameter.
   * ParameterList)
   */
  @Override
  public void setParameters(ParameterList params) {
    this.parameters.setParameters(params);
  }
}
