/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.policy;

import de.fraunhofer.iese.ind2uce.api.common.Ind2uceEntity;
import de.fraunhofer.iese.ind2uce.api.policy.parameter.Parameter;
import de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterList;
import de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider;

import org.apache.commons.lang3.builder.HashCodeBuilder;

/**
 * A modifier engine describes a method that can be performed on event
 * parameters. Despite an engine name it contains a list of parameters required
 * for the processing the engine.
 *
 * @author Fraunhofer IESE
 */
public class ModifierEngine extends Ind2uceEntity implements ParameterListProvider {

  /**
   * The Constant serialVersionUID.
   */
  private static final long serialVersionUID = 4050073698831680651L;

  /**
   * The engine name.
   */
  private String method;

  /**
   * List of parameters modifiers for this modifier engine; use map here to
   * guarantee, that every parameter can only exist once!.
   */
  // private List<Param<?>> params = new ArrayList<>();
  private final ParameterList parameters = new ParameterList();

  /**
   * Instantiates a new modifier engine.
   *
   * @param name the name of the parameter
   */
  public ModifierEngine(String name) {
    this(name, null);
  }

  /**
   * Instantiates a new modifier engine.
   *
   * @param name the name of the parameter
   * @param params the parameter values
   */
  public ModifierEngine(String name, ParameterList params) {
    this.method = name;
    this.setParameters(params);
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * addParameter(de.fraunhofer.iese.ind2uce.api.policy.parameter.Parameter)
   */
  @Override
  public void addParameter(Parameter<?> param) {
    this.parameters.add(param);
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * addParameter(java.lang.String, java.lang.Object)
   */
  @Override
  public <T> void addParameter(String name, T value) {
    this.parameters.add(new Parameter<>(name, value));
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * clearParameters()
   */
  @Override
  public void clearParameters() {
    this.parameters.clear();
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  public boolean equals(Object obj) {
    if (super.equals(obj) == false || !(obj instanceof ModifierEngine)) {
      return false;
    }

    final ModifierEngine other = (ModifierEngine)obj;

    if (this.method == null && other.getMethod() != null) {
      return false;
    }

    return this.method.equals(other.getMethod());
  }

  /**
   * Gets the engine name.
   *
   * @return the engine name
   */
  public String getMethod() {
    return this.method;
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * getParameterForName(java.lang.String)
   */
  @Override
  public Parameter<?> getParameterForName(String name) {
    return this.parameters.getParameterForName(name);
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * getParameters()
   */
  @Override
  public ParameterList getParameters() {
    return this.parameters;
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * getParameterValue(java.lang.String, java.lang.Class)
   */
  @Override
  public <T> T getParameterValue(String name, Class<T> clazz) {
    return this.parameters.getParameterValue(name, clazz);
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#hashCode()
   */
  @Override
  public int hashCode() {
    final HashCodeBuilder builder = new HashCodeBuilder(17, 31);
    builder.append(super.hashCode());
    builder.append(this.getMethod());
    return builder.toHashCode();
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * removeParameter(java.lang.String)
   */
  @Override
  public void removeParameter(String name) {
    this.parameters.remove(name);
  }

  /**
   * Sets the engine name.
   *
   * @param method the engine name to set
   */
  public void setMethod(String method) {
    this.method = method;
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * setParameters(de.fraunhofer.iese.ind2uce.api.policy.parameter.
   * ParameterList)
   */
  @Override
  public void setParameters(ParameterList params) {
    this.parameters.setParameters(params);
  }

}
