/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.policy;

import java.util.ArrayList;
import java.util.List;

/**
 * A modifier describes a method that can be performed on events. Despite a key
 * and value, it also contains an engine.
 *
 * @author Fraunhofer IESE
 */
// TODO change the name to EventParameter as PDP
public class Modifier {

  /**
   * The Constant serialVersionUID.
   */
  private static final long serialVersionUID = 931802052960953294L;

  /**
   * The name.
   */
  private String name;

  /**
   * The JsonPath expression.
   */
  private String expression;

  /**
   * An Engine used to modify the parameter value.
   */
  private List<ModifierEngine> engine;

  /**
   * Instantiates a new modifier.
   *
   * @param name the name
   */
  public Modifier(String name) {
    this.name = name;
  }

  /**
   * Instantiates a new modifier.
   *
   * @param name the name of the parameter
   * @param engine the engine to be used
   */
  public Modifier(String name, List<ModifierEngine> engine) {
    this.setName(name);
    this.engine = engine;
  }

  /**
   * Adds the engine.
   *
   * @param engine the engine to set
   */
  public void addEngine(ModifierEngine engine) {
    if (this.engine == null) {
      this.engine = new ArrayList<>();
    }
    this.engine.add(engine);
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (!(o instanceof Modifier)) {
      return false;
    }

    final Modifier modifier = (Modifier)o;

    if (this.name != null ? !this.name.equals(modifier.name) : modifier.name != null) {
      return false;
    }
    return this.engine != null ? this.engine.equals(modifier.engine) : modifier.engine == null;

  }

  /**
   * Gets the an Engine used to modify the parameter value.
   *
   * @return the current modifier engine
   */
  public List<ModifierEngine> getEngine() {
    return this.engine;
  }

  /**
   * Gets the expression.
   *
   * @return the expression
   */
  public String getExpression() {
    return this.expression;
  }

  /**
   * Gets the name.
   *
   * @return the name
   */
  public String getName() {
    return this.name;
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#hashCode()
   */
  @Override
  public int hashCode() {
    int result = this.name != null ? this.name.hashCode() : 0;
    result = 31 * result + (this.engine != null ? this.engine.hashCode() : 0);
    return result;
  }

  /**
   * Sets the an Engine used to modify the parameter value.
   *
   * @param engine the engine to set
   */
  public void setEngine(List<ModifierEngine> engine) {
    this.engine = engine;
  }

  /**
   * Sets the expression.
   *
   * @param expression the expression to set
   */
  public void setExpression(String expression) {
    this.expression = expression;
  }

  /**
   * Sets the name.
   *
   * @param name the name to set
   */
  public void setName(String name) {
    this.name = name;
  }
}
