/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.policy;

import de.fraunhofer.iese.ind2uce.api.common.Ind2uceEntity;
import de.fraunhofer.iese.ind2uce.api.policy.identifier.ActionId;
import de.fraunhofer.iese.ind2uce.api.policy.parameter.Parameter;
import de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterList;
import de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider;
import de.fraunhofer.iese.ind2uce.logger.LoggerFactory;

import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.slf4j.Logger;

/**
 * An action to be performed. Actions have a name and a set of name-value
 * parameters.
 */
public class ExecuteAction extends Ind2uceEntity implements ParameterListProvider {

  /**
   * The Constant serialVersionUID.
   */
  private static final long serialVersionUID = -8106817237376519851L;

  /**
   * The Constant LOG.
   */
  private static final Logger LOG = LoggerFactory.getLogger(ExecuteAction.class);

  /**
   * The id.
   */
  private ActionId id;

  /**
   * The parameters.
   */
  private final ParameterList parameters = new ParameterList();

  /**
   * Instantiates a new execute action.
   */
  public ExecuteAction() {
  }

  /**
   * Instantiates a new execute action.
   *
   * @param actionId the action id
   * @param params the params
   */
  public ExecuteAction(ActionId actionId, ParameterList params) {
    this.id = actionId;
    this.setParameters(params);
  }

  /**
   * Instantiates a new execute action.
   *
   * @param actionId the action id
   */
  public ExecuteAction(String actionId) {
    this(new ActionId(actionId), null);
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * addParameter(de.fraunhofer.iese.ind2uce.api.policy.parameter.Parameter)
   */
  @Override
  public void addParameter(Parameter<?> param) {
    this.parameters.add(param);
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * addParameter(java.lang.String, java.lang.Object)
   */
  @Override
  public <T> void addParameter(String name, T value) {
    this.parameters.add(new Parameter<>(name, value));
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * clearParameters()
   */
  @Override
  public void clearParameters() {
    this.parameters.clear();
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  public boolean equals(Object obj) {
    if (obj == null) {
      return false;
    }
    if (!(obj instanceof ExecuteAction)) {
      return false;
    }

    final ExecuteAction exec2 = (ExecuteAction)obj;
    LOG.debug("Comparing executeAction: " + this.getId());

    if (!this.id.equals(exec2.getId())) {
      return false;
    }

    if (this.parameters.size() != exec2.parameters.size()) {
      return false;
    }

    for (final Parameter<?> curParam : this.parameters) {
      LOG.debug("Comparing param: " + curParam.getName());
      if (!curParam.equals(exec2.getParameterForName(curParam.getName()))) {
        LOG.debug("param " + curParam.getName() + " differ.");
        return false;
      }
    }

    return true;
  }

  /**
   * Gets the id.
   *
   * @return the id
   */
  public ActionId getId() {
    return this.id;
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * getParameterForName(java.lang.String)
   */
  @Override
  public Parameter<?> getParameterForName(String name) {
    return this.parameters.getParameterForName(name);
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * getParameters()
   */
  @Override
  public ParameterList getParameters() {
    return this.parameters;
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * getParameterValue(java.lang.String, java.lang.Class)
   */
  @Override
  public <T> T getParameterValue(String name, Class<T> clazz) {
    return this.parameters.getParameterValue(name, clazz);
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#hashCode()
   */
  @Override
  public int hashCode() {
    final HashCodeBuilder builder = new HashCodeBuilder(17, 31);
    builder.append(this.id);
    for (final Parameter<?> p : this.parameters) {
      builder.append(p.getName());
      builder.append(p.getValue());
    }
    return builder.toHashCode();
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * removeParameter(java.lang.String)
   */
  @Override
  public void removeParameter(String name) {
    this.parameters.remove(name);
  }

  /**
   * Sets the id.
   *
   * @param id the new id
   */
  public void setId(ActionId id) {
    this.id = id;
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * setParameters(de.fraunhofer.iese.ind2uce.api.policy.parameter.
   * ParameterList)
   */
  @Override
  public void setParameters(ParameterList params) {
    this.parameters.setParameters(params);
  }
}
