/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.policy;

import de.fraunhofer.iese.ind2uce.api.common.Ind2uceEntity;
import de.fraunhofer.iese.ind2uce.api.policy.identifier.ActionId;
import de.fraunhofer.iese.ind2uce.api.policy.parameter.Parameter;
import de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterList;
import de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider;

/**
 * An event that occured in the system under observation, intercepted or
 * monitored by a PolicyEnforcementPoint.
 *
 * @author Fraunhofer IESE
 */
public class Event extends Ind2uceEntity implements ParameterListProvider {

  /**
   * The Constant serialVersionUID.
   */
  private static final long serialVersionUID = -1664098616473093266L;

  /**
   * The id of the system action that occurred.
   */
  private ActionId actionId;

  /**
   * The time at which the action occurred / was intercepted by the PEP.
   */
  private long timestamp;

  /**
   * * Will be used by solution developer to identify event uniquely.
   */
  private transient Object tag;

  /**
   * The parameters.
   */
  private ParameterList parameters = new ParameterList();

  /**
   * Instantiates a new event.
   *
   * @param actionId the action id
   * @param isTry the is try
   * @param time the time
   * @param params the params
   */
  public Event(ActionId actionId, boolean isTry, long time, Parameter<?>... params) {
    this(actionId, isTry, time, new ParameterList(params));
  }

  /**
   * Instantiates a new event.
   *
   * @param actionId the action id
   * @param isTry the is try
   * @param time the time
   * @param params the params
   */
  public Event(ActionId actionId, boolean isTry, long time, ParameterList params) {
    if (actionId == null) {
      throw new IllegalArgumentException("ActionId may not be null");
    }
    this.actionId = actionId;
    this.timestamp = time;
    this.setParameters(params);
  }

  /**
   * Instantiates a new event.
   *
   * @param actionId the action id
   * @param isTry the is try
   * @param params the params
   */
  public Event(ActionId actionId, boolean isTry, Parameter<?>... params) {
    this(actionId, isTry, System.currentTimeMillis(), params);
  }

  /**
   * Instantiates a new event.
   *
   * @param actionId the action id
   * @param params the params
   */
  public Event(ActionId actionId, Parameter<?>... params) {
    this(actionId, false, System.currentTimeMillis(), params);
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * addParameter(de.fraunhofer.iese.ind2uce.api.policy.parameter.Parameter)
   */
  @Override
  public void addParameter(Parameter<?> param) {
    if (this.parameters == null) {
      this.parameters = new ParameterList();
    }
    this.parameters.add(param);
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * addParameter(java.lang.String, java.lang.Object)
   */
  @Override
  public <T> void addParameter(String name, T value) {
    if (this.parameters == null) {
      this.parameters = new ParameterList();
    }
    this.parameters.add(new Parameter<>(name, value));
  }

  /**
   * Clear parameters.
   */
  @Override
  public void clearParameters() {
    this.parameters.clear();
  }

  /**
   * Gets the id of the system action that occurred.
   *
   * @return the ID of the event
   */
  public ActionId getActionId() {
    return this.actionId;
  }

  /**
   * Gets the action id string.
   *
   * @return the action id string
   */
  public String getActionIdString() {
    return this.actionId.toString();
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * getParameterForName(java.lang.String)
   */
  @Override
  public Parameter<?> getParameterForName(String name) {
    if (this.parameters == null) {
      return null;
    }
    return this.parameters.getParameterForName(name);
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * getParameters()
   */
  @Override
  public ParameterList getParameters() {
    return this.parameters;
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * getParameterValue(java.lang.String, java.lang.Class)
   */
  @Override
  public <T> T getParameterValue(String name, Class<T> clazz) {
    if (this.parameters == null) {
      return null;
    }
    return this.parameters.getParameterValue(name, clazz);
  }

  /**
   * *.
   *
   * @return the * Will be used by solution developer to identify event uniquely
   */
  public Object getTag() {
    return this.tag;
  }

  /**
   * Gets the time at which the action occurred / was intercepted by the PEP.
   *
   * @return the timestamp
   */
  public long getTimestamp() {
    return this.timestamp;
  }

  /**
   * Gets the object which is added as parameter for given name.
   *
   * @param name the name
   * @return the object
   */
  public Object getValueForName(String name) {
    final Parameter<?> param = this.getParameterForName(name);
    return param.getValue();
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * removeParameter(java.lang.String)
   */
  @Override
  public void removeParameter(String name) {
    this.parameters.remove(this.getParameterForName(name));
  }

  /**
   * Sets the id of the system action that occurred.
   *
   * @param actionId the new id of the system action that occurred
   */
  public void setActionId(ActionId actionId) {
    this.actionId = actionId;
  }

  /**
   * Sets the id of the system action that occurred.
   *
   * @param actionId the new id of the system action that occurred
   */
  public void setActionId(String actionId) {
    this.actionId = new ActionId(actionId);
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterListProvider#
   * setParameters(de.fraunhofer.iese.ind2uce.api.policy.parameter.
   * ParameterList)
   */
  @Override
  public void setParameters(ParameterList params) {
    if (this.parameters == null) {
      this.parameters = new ParameterList();
    }
    this.parameters.setParameters(params);
  }

  /**
   * *.
   *
   * @param tag the new * Will be used by solution developer to identify event
   *          uniquely
   */
  public void setTag(Object tag) {
    this.tag = tag;
  }

  /**
   * Sets the time at which the action occurred / was intercepted by the PEP.
   *
   * @param timestamp the timestamp to set
   */
  public void setTimestamp(long timestamp) {
    this.timestamp = timestamp;
  }
}
