/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.policy;

import de.fraunhofer.iese.ind2uce.api.common.Ind2uceEntity;
import de.fraunhofer.iese.ind2uce.api.policy.identifier.DecisionId;
import de.fraunhofer.iese.ind2uce.api.policy.parameter.ModifierList;

import java.util.List;

/**
 * Information about the permissiveness of the event. The decision may include
 * instructions for modification or delaying the event.Decisions are provided by
 * the PDP and enforced by a PEP.
 *
 * @author Fraunhofer IESE
 */
public class AuthorizationDecision extends Ind2uceEntity {

  /**
   * The Constant serialVersionUID.
   */
  private static final long serialVersionUID = -6800258958385977399L;

  /**
   * The default decision for a plain allow.
   */
  public static final AuthorizationDecision DECISION_ALLOW = new AuthorizationDecision(new DecisionId("urn:decision:allow"), true);

  /**
   * The default decision for a plain inhibit.
   */
  public static final AuthorizationDecision DECISION_INHIBIT = new AuthorizationDecision(new DecisionId("urn:decision:inhibit"), false);

  /**
   * The id of the decision.
   */
  private DecisionId id;

  /**
   * Indicates whether an event should be allowed or blocked.
   */
  private boolean eventAllowed;

  /**
   * The name(s) of the corresponding AutorizationAction(s) that led to the
   * decision.
   */
  private List<String> authorizationActionNames;

  /**
   * List of modifications that need to be enforced for an event.
   */
  private final ModifierList modifiers = new ModifierList();

  /**
   * The amount of microseconds the event should be delayed.
   */
  private long delay = 0;

  /**
   * Instantiates a new authorization decision.
   */
  public AuthorizationDecision() {
  }

  /**
   * Constructor that takes the id and eventAllowed from Prototype.
   *
   * @param prototype Prototype to create a new instance.
   */
  public AuthorizationDecision(AuthorizationDecision prototype) {
    this(prototype.getId(), prototype.isEventAllowed());
  }

  /**
   * Instantiates a new authorization decision.
   *
   * @param id the id
   * @param allowEvent the allow event
   */
  private AuthorizationDecision(DecisionId id, boolean allowEvent) {
    this(id, allowEvent, 0, null);
  }

  /**
   * Instantiates a new authorization decision.
   *
   * @param id the id
   * @param eventAllowed the event allowed
   * @param delay the delay unit
   * @param authorizationActionNames the authorization action names
   * @param modifiers the modifiers
   */
  public AuthorizationDecision(DecisionId id, boolean eventAllowed, long delay, List<String> authorizationActionNames, Modifier... modifiers) {
    this(id, eventAllowed, delay, authorizationActionNames, new ModifierList(modifiers));
  }

  /**
   * Instantiates a new authorization decision.
   *
   * @param id the id
   * @param eventAllowed the event allowed
   * @param delay the delay unit
   * @param authorizationActionNames the authorization action names
   * @param modifiers the modifiers
   */
  public AuthorizationDecision(DecisionId id, boolean eventAllowed, long delay, List<String> authorizationActionNames, ModifierList modifiers) {
    this.id = id;
    this.eventAllowed = eventAllowed;
    if (authorizationActionNames != null) {
      this.setAuthorizationActionNames(authorizationActionNames);
    }
    if (eventAllowed) {
      this.setModifiers(modifiers);
      this.setDelay(delay);
    }
  }

  /**
   * Get a clone of DECISION_ALLOW.
   *
   * @return a clone of DECISION_ALLOW.
   */
  public static final AuthorizationDecision getDecisionAllow() {
    return new AuthorizationDecision(DECISION_ALLOW);

  }

  /**
   * Get a clone of DECISION_INHIBIT.
   *
   * @return DECISION_INHIBIT
   */
  public static final AuthorizationDecision getDecisionInhibit() {
    return new AuthorizationDecision(DECISION_INHIBIT);

  }

  /**
   * Adds the modifier.
   *
   * @param param the param
   */
  public void addModifier(Modifier param) {
    this.modifiers.add(param);
  }

  /**
   * Clear modifiers.
   */
  public void clearParameters() {
    this.modifiers.clear();
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (!(o instanceof AuthorizationDecision)) {
      return false;
    }

    final AuthorizationDecision that = (AuthorizationDecision)o;

    if (this.eventAllowed != that.eventAllowed) {
      return false;
    }
    if (this.delay != that.delay) {
      return false;
    }
    if (this.id != null ? !this.id.equals(that.id) : that.id != null) {
      return false;
    }
    if (this.authorizationActionNames != null ? !this.authorizationActionNames.equals(that.authorizationActionNames) : that.authorizationActionNames != null) {
      return false;
    }
    return !(this.modifiers != null ? !this.modifiers.equals(that.modifiers) : that.modifiers != null);

  }

  /**
   * Gets the name(s) of the corresponding AutorizationAction(s) that led to the
   * decision.
   *
   * @return the name(s) of the corresponding AutorizationAction(s) that led to
   *         the decision
   */
  public List<String> getAuthorizationActionNames() {
    return this.authorizationActionNames;
  }

  /**
   * Returns how much the event should be delayed.
   *
   * @return the delay in microseconds
   */
  public long getDelay() {
    return this.delay;
  }

  /**
   * Gets the id.
   *
   * @return the id
   */
  public DecisionId getId() {
    return this.id;
  }

  /**
   * Gets the modifier for name.
   *
   * @param name the name
   * @return the modifier for name
   */
  public Modifier getModifierForName(String name) {
    return this.modifiers.getParameterForName(name);
  }

  /**
   * Gets the modifiers. Parameters in this list can be of type {@link Modifier}
   * .
   *
   * @return modifiers the list of modifications that need to be enforced for an
   *         event.
   */
  public ModifierList getModifiers() {
    return this.modifiers;
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#hashCode()
   */
  @Override
  public int hashCode() {
    int result = this.id != null ? this.id.hashCode() : 0;
    result = 31 * result + (this.eventAllowed ? 1 : 0);
    result = 31 * result + (this.authorizationActionNames != null ? this.authorizationActionNames.hashCode() : 0);
    result = 31 * result + (this.modifiers != null ? this.modifiers.hashCode() : 0);
    result = 31 * result + (int)(this.delay ^ (this.delay >>> 32));
    return result;
  }

  /**
   * Indicates whether an event should be allowed or blocked.
   *
   * @return true, if the event should be allowed, false otherwise
   */
  public boolean isEventAllowed() {
    return this.eventAllowed;
  }

  /**
   * Removes the modifier.
   *
   * @param name the name
   */
  public void removeParameter(String name) {
    this.modifiers.remove(name);
  }

  /**
   * Sets the authorization action names.
   *
   * @param authorizationActionNames the names of the authorizationActions that
   *          led to the decision.
   */
  public void setAuthorizationActionNames(List<String> authorizationActionNames) {
    this.authorizationActionNames = authorizationActionNames;
  }

  /**
   * Sets how much the event should be delayed.
   *
   * @param delay the delay in microseconds
   */
  public void setDelay(long delay) {
    this.delay = delay;
  }

  /**
   * Sets whether an event should be allowed or blocked.
   *
   * @param eventAllowed true, if the event should be allowed, false otherwise
   */
  public void setEventAllowed(boolean eventAllowed) {
    this.eventAllowed = eventAllowed;
  }

  /**
   * Sets the id.
   *
   * @param id the new id
   */
  public void setId(DecisionId id) {
    this.id = id;
  }

  /**
   * Sets modifiers for the event. Note that modifiers are possible if the event
   * is allowed.
   *
   * @param modifiers the new modifiers
   */
  public void setModifiers(ModifierList modifiers) {
    if (this.eventAllowed) {
      this.clearParameters();
      this.modifiers.addAll(modifiers);
    }
  }
}
