/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.component.interfaces;

import de.fraunhofer.iese.ind2uce.api.component.Component;
import de.fraunhofer.iese.ind2uce.api.component.EnforcementScope;
import de.fraunhofer.iese.ind2uce.api.component.identifier.ComponentId;
import de.fraunhofer.iese.ind2uce.api.component.identifier.EnforcementScopeId;
import de.fraunhofer.iese.ind2uce.api.component.identifier.EnforcementScopeTypeId;
import de.fraunhofer.iese.ind2uce.api.component.identifier.GroupId;
import de.fraunhofer.iese.ind2uce.api.policy.identifier.PolicyId;
import de.fraunhofer.iese.ind2uce.api.policy.parameter.Parameter;

import java.io.IOException;
import java.net.URI;
import java.util.List;

/**
 * Server Interface of the Policy Management Point. It is mainly responsible to
 * manage enforcement scopes and Client PMPs.
 *
 * @author Fraunhofer IESE
 */
public interface IRootPolicyManagementPoint extends IComponent {

  /**
   * Adds an alias for a given enforcement scope.
   *
   * @param enforcementScope the enforcement scope
   * @param alias an alias for the enforcement scope (e.g., an IP address)
   * @return true, if the alias was added, false otherwise
   * @throws IOException communication failure
   */
  boolean addAlias(EnforcementScopeId enforcementScope, String alias) throws IOException;

  /**
   * Adds an enforcement scope to a group. If the group does not exist, it will
   * be created.
   *
   * @param groupId the target group
   * @param enforcementScope the ID of the enforcement scope to be added
   * @return true, if added successfully, false otherwise
   * @throws IOException communication failure
   */
  boolean addToGroup(GroupId groupId, EnforcementScopeId enforcementScope) throws IOException;

  /**
   * Adds a URL to an already registered client PMP.
   *
   * @param componentId the ID of the PMP
   * @param uri the URI to add
   * @return true, if the URL was added, false otherwise
   * @throws IOException communication failure
   */
  boolean addURI(ComponentId componentId, String uri) throws IOException;

  /**
   * Deploys a policy at the PDP of the enforcement scope.
   *
   * @param enforcementScope the target system
   * @param policyId the id of the policy to be deployed. Will be retrieved via
   *          the PRP
   * @param policyParameters a list of parameters that is used to instantiate
   *          policy templates
   * @return the {@link PolicyId} of the deployed policy, or null if the
   *         operation fails
   * @throws IOException communication failure
   */
  PolicyId deployPolicy(EnforcementScopeId enforcementScope, PolicyId policyId, List<Parameter<String>> policyParameters) throws IOException;

  /**
   * Deploys a policy at all enforcement scopes of the given group.
   *
   * @param groupId the ID of the target group
   * @param policyId the id of the policy to be deployed. Will be retrieved via
   *          the PRP
   * @param policyParameters a list of parameters that is used to instantiate
   *          policy templates
   * @param keepWhenLeaving when set to false, the policy will be revoked when
   *          an enforcement scope leaves the group
   * @return the {@link PolicyId} of the deployed policy, or null if the
   *         operation fails
   * @throws IOException communication failure
   */
  PolicyId deployPolicy(GroupId groupId, PolicyId policyId, List<Parameter<String>> policyParameters, boolean keepWhenLeaving) throws IOException;

  /**
   * Deploys a policy at the PDP of the enforcement scope.
   *
   * @param alias the alias (e.g., IP or MAC address) target system
   * @param policyId the id of the policy to be deployed. Will be retrieved via
   *          the PRP
   * @param policyParameters a list of parameters that is used to instantiate
   *          policy templates
   * @return the {@link PolicyId} of the deployed policy, or null if the
   *         operation fails
   * @throws IOException communication failure
   */
  PolicyId deployPolicy(String alias, PolicyId policyId, List<Parameter<String>> policyParameters) throws IOException;

  /**
   * Deploys a policy at all enforcement scopes that match the regular
   * expression.
   *
   * @param expression an expression that matches groups or enforcement scopes
   * @param policyId the id of the policy to be deployed. Will be retrieved via
   *          the PRP
   * @param policyParameters a list of parameters that is used to instantiate
   *          policy templates
   * @param keepWhenLeaving when set to false, the policy will be revoked when
   *          an enforcement scope does not match the expression
   * @return the {@link PolicyId} of the deployed policy, or null if the
   *         operation fails
   * @throws IOException communication failure
   */
  PolicyId deployPolicy(String expression, PolicyId policyId, List<Parameter<String>> policyParameters, boolean keepWhenLeaving) throws IOException;

  /**
   * Resolves all enforcement scopes of a certain group.
   *
   * @param group the group ID
   * @return a list of enforcement scopes that belong to the given group
   * @throws IOException communication failure
   */
  List<EnforcementScopeId> getEnforcementScopes(GroupId group) throws IOException;

  /**
   * Resolves all enforcement scopes that match the regular expression.
   *
   * @param expression a regular expression that is matched against the
   *          {@link EnforcementScopeId}
   * @return a list of enforcement scopes that match the expression
   * @throws IOException communication failure
   */
  List<EnforcementScopeId> getEnforcementScopes(String expression) throws IOException;

  /**
   * Resolves a PMP.
   *
   * @param enforcementScope the ID of the target enforcement scope
   * @return a map of client PMPs and corresponding URLs. Size can be 0 or 1.
   * @throws IOException communication failure
   */
  List<Component> lookupPmp(EnforcementScopeId enforcementScope) throws IOException;

  /**
   * Resolves all client PMPs of a certain group.
   *
   * @param group the group ID
   * @return a list of PMPs that belong to the given group
   * @throws IOException communication failure
   */
  List<Component> lookupPmpByGroup(GroupId group) throws IOException;

  /**
   * Resolves all PMPs that match the regular expression.
   *
   * @param expression a regular expression that is matched against the
   *          {@link ComponentId}
   * @return a list of PMPs that match the expression
   * @throws IOException communication failure
   */
  List<Component> lookupPmpByGroup(String expression) throws IOException;

  /**
   * Registers a client PMP and an enforcement scope.
   *
   * @param componentId the ID of the PMP of the enforcement scope
   * @param enforcementScope the ID of the enforcement scope
   * @param esType the type of the enforcement scope
   * @param urls the urls
   * @param aliases a list of aliases for the enforcement scope
   * @return true, if the enforcement scope was added, false otherwise
   * @throws IOException communication failure
   */
  boolean registerPmp(ComponentId componentId, EnforcementScopeId enforcementScope, EnforcementScopeTypeId esType, List<URI> urls, List<String> aliases) throws IOException;

  /**
   * Removes an alias for a given enforcement scope.
   *
   * @param enforcementScope the enforcement scope
   * @param alias the alias for the enforcement scope (e.g., an IP address) that
   *          should be removed
   * @return true, if the alias was removed, false otherwise
   * @throws IOException communication failure
   */
  boolean removeAlias(EnforcementScopeId enforcementScope, String alias) throws IOException;

  /**
   * Removes an enforcement scope from a group.
   *
   * @param groupId the target group
   * @param enforcementScope the ID of the enforcement scope to be removed
   * @return true, if removed successfully, false otherwise
   * @throws IOException communication failure
   */
  boolean removeFromGroup(GroupId groupId, EnforcementScopeId enforcementScope) throws IOException;

  /**
   * Removes a URL of an already registered client PMP.
   *
   * @param componentId the ID of the PMP
   * @param uri the URI to remove
   * @return true, if the URL was removed, false otherwise
   * @throws IOException communication failure
   */
  boolean removeURI(ComponentId componentId, String uri) throws IOException;

  /**
   * Removed a PMP and enforcement scope.
   *
   * @param componentId the ID of the client PMP
   * @return true, if the enforcement scope was removed, false otherwise
   * @throws IOException communication failure
   */
  boolean unregisterPmp(ComponentId componentId) throws IOException;

  /**
   * Updates the type of the given enforcement scope.
   *
   * @param enforcementScope the ID of the enforcement scope
   * @param type the new type ID
   * @param desc the new type
   * @return true if the type has been updated, false otherwise
   * @throws IOException communication failure
   */
  boolean updateType(EnforcementScopeId enforcementScope, EnforcementScopeTypeId type, EnforcementScope desc) throws IOException;

}
