/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.component.interfaces;

import de.fraunhofer.iese.ind2uce.api.component.Component;
import de.fraunhofer.iese.ind2uce.api.component.ComponentBase;
import de.fraunhofer.iese.ind2uce.api.component.PepComponent;
import de.fraunhofer.iese.ind2uce.api.component.description.MethodInterfaceDescription;
import de.fraunhofer.iese.ind2uce.api.component.identifier.ComponentId;
import de.fraunhofer.iese.ind2uce.api.policy.Policy;
import de.fraunhofer.iese.ind2uce.api.policy.identifier.PolicyId;

import java.io.IOException;
import java.util.List;

/**
 * Policy Retrieval Points provides data storage for the PMP.
 *
 * @author Fraunhofer IESE
 */
public interface IPolicyRetrievalPoint extends IComponent {

  /**
   * Adds a policy (ILP, ILPT, SLP, SLPT).
   *
   * @param policy the policy
   * @return true if successfully added
   * @throws IOException Signals that an I/O exception has occurred.
   */
  boolean addPolicy(Policy policy) throws IOException;

  /**
   * Checks if the component is already in the database.
   *
   * @param componentId the id of the component
   * @return true if the component is already registered, false otherwise
   * @throws IOException communication failure
   */
  boolean componentExists(ComponentId componentId) throws IOException;

  /**
   * Gets all policy.
   *
   * @return all policy
   * @throws IOException Signals that an I/O exception has occurred.
   */
  List<Policy> getDeployedPolicies() throws IOException;

  /**
   * Gets the deployed policies.
   *
   * @param solutionId the solution id
   * @return the deployed policies
   */
  List<Policy> getDeployedPolicies(String solutionId);

  /**
   * Gets the status.
   *
   * @param policyId the policy id
   * @return the status
   * @throws IOException Signals that an I/O exception has occurred.
   */
  boolean getDeploymentStatus(PolicyId policyId) throws IOException;

  /**
   * Gets all policy.
   *
   * @return all policy
   * @throws IOException Signals that an I/O exception has occurred.
   */
  List<Policy> getPolicies() throws IOException;

  /**
   * Gets the policies.
   *
   * @param solutionId the solution id
   * @return the policies
   */
  List<Policy> getPolicies(String solutionId);

  /**
   * Gets the latest policy.
   *
   * @param policyId the policy id
   * @return the latest policy
   * @throws IOException Signals that an I/O exception has occurred.
   */
  Policy getPolicy(PolicyId policyId) throws IOException;

  /**
   * Gets all policy ids.
   *
   * @return all policy IDs
   * @throws IOException Signals that an I/O exception has occurred.
   */
  List<PolicyId> getPolicyIds() throws IOException;

  /**
   * Provides a list of currently deployed policy ids.
   *
   * @return list of currently deployed policy ids
   * @throws IOException communication failure
   */
  List<PolicyId> listDeployedPolicies() throws IOException;

  /**
   * Lookup by {@link ComponentId}.
   *
   * @param id the id
   * @return the component
   * @throws IOException communication failure
   */
  ComponentBase lookup(ComponentId id) throws IOException;

  /**
   * Returns the list of PDPs in the database.
   *
   * @return the PDP in the database
   * @throws IOException communication failure
   */
  Component lookupPdp() throws IOException;

  /**
   * Returns the list of PEPs in the database.
   *
   * @return the list of PEPs in the database
   * @throws IOException communication failure
   */
  List<PepComponent> lookupPep() throws IOException;

  /**
   * Lookup by {@link ComponentId}.
   *
   * @param id the id
   * @return the component
   * @throws IOException communication failure
   */
  PepComponent lookupPep(ComponentId id) throws IOException;

  /**
   * Returns the list of PIPs in the database.
   *
   * @return the list of PIPs in the database
   * @throws IOException communication failure
   */
  List<Component> lookupPip() throws IOException;

  /**
   * Returns the list of PIPs in the database.
   *
   * @param query the required / available {@link MethodInterfaceDescription}s
   *          to match against
   * @return the list of PIPs in the database
   * @throws IOException communication failure
   */
  List<Component> lookupPip(MethodInterfaceDescription query) throws IOException;

  /**
   * Returns the list of PXPs in the database.
   *
   * @return the list of PXPs in the database
   * @throws IOException communication failure
   */
  List<Component> lookupPxp() throws IOException;

  /**
   * Returns the list of PXPs in the database.
   *
   * @param query the required / available {@link MethodInterfaceDescription}s
   *          to match against
   * @return the list of PXPs in the database
   * @throws IOException communication failure
   */
  List<Component> lookupPxp(MethodInterfaceDescription query) throws IOException;

  /**
   * Adds a component into the database.
   *
   * @param component the component
   * @return true if the component was successfully added, false otherwise
   * @throws IOException communication failure
   */
  boolean registerComponent(ComponentBase component) throws IOException;

  /**
   * Adds a pep into the database.
   *
   * @param component the component
   * @return true if the component was successfully added, false otherwise
   * @throws IOException communication failure
   */
  boolean registerPEPComponent(PepComponent component) throws IOException;

  /**
   * Removes the policy.
   *
   * @param policy the policy
   * @param reason the reason
   * @return true, if successful
   * @throws IOException Signals that an I/O exception has occurred.
   */
  boolean removePolicy(PolicyId policy, String reason) throws IOException;

  /**
   * Sets the status.
   *
   * @param policyId the policy id
   * @param deployed the deployed
   * @throws IOException Signals that an I/O exception has occurred.
   */
  void setDeploymentStatus(PolicyId policyId, boolean deployed) throws IOException;

  /**
   * Deletes a component from the database.
   *
   * @param componentId the ID of the component to be removed
   * @return true if removed successfully (or did not exist), false otherwise
   * @throws IOException communication failure
   */
  boolean unregisterComponent(ComponentId componentId) throws IOException;

  /**
   * Unregisters a pep by id.
   *
   * @param componentId the componentId
   * @return true if the componentId was successfully removed, false otherwise
   * @throws IOException communication failure
   */
  boolean unregisterPEPComponent(ComponentId componentId) throws IOException;

}
