/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.component.interfaces;

import de.fraunhofer.iese.ind2uce.api.component.Component;
import de.fraunhofer.iese.ind2uce.api.component.ComponentBase;
import de.fraunhofer.iese.ind2uce.api.component.EnforcementScope;
import de.fraunhofer.iese.ind2uce.api.component.PepComponent;
import de.fraunhofer.iese.ind2uce.api.component.description.MethodInterfaceDescription;
import de.fraunhofer.iese.ind2uce.api.component.exception.ConflictingPolicyException;
import de.fraunhofer.iese.ind2uce.api.component.exception.PolicyRevokationException;
import de.fraunhofer.iese.ind2uce.api.component.identifier.ComponentId;
import de.fraunhofer.iese.ind2uce.api.component.identifier.EnforcementScopeId;
import de.fraunhofer.iese.ind2uce.api.policy.InvalidPolicyException;
import de.fraunhofer.iese.ind2uce.api.policy.Policy;
import de.fraunhofer.iese.ind2uce.api.policy.identifier.PolicyId;

import java.io.IOException;
import java.util.List;
import java.util.NoSuchElementException;

// TODO: Auto-generated Javadoc
/**
 * Client Interface of the Policy Management Point. It is mainly responsible to
 * manage local components and to deploy policies at the PDP of the enforcement
 * scope.
 *
 * @author Fraunhofer IESE
 */
public interface IPolicyManagementPoint extends IComponent {

  /**
   * Deploys a policy at the local PDP of the enforcement scope. The policy will
   * be stored at the PRP with the given ID.
   *
   * @param policy the policy.
   * @return true, if the policy was deployed successfully, false otherwise
   * @throws IOException if connection problem occurs
   * @throws IllegalArgumentException policy is not valid
   * @throws ConflictingPolicyException multiple policies with same policyId
   *           exist in PMP database
   * @throws InvalidPolicyException
   */
  boolean addAndDeployPolicy(String policy) throws IOException, IllegalArgumentException, ConflictingPolicyException, InvalidPolicyException;

  /**
   * Adds a policy (ILP, ILPT, SLP, SLPT).
   *
   * @param policy the policy
   * @return true if successfully added
   * @throws IOException Signals that an I/O exception has occurred.
   * @throws InvalidPolicyException
   */
  boolean addPolicyString(String policy) throws IOException;

  /**
   * Adds a policy (ILP, ILPT, SLP, SLPT).
   *
   * @param policy the policy
   * @return true if successfully added
   * @throws IOException Signals that an I/O exception has occurred.
   * @throws InvalidPolicyException
   */
  boolean addPolicy(Policy p) throws IOException;

  /**
   * Checks if the component is already in the database.
   *
   * @param componentId the id of the component
   * @return true if the component is already registered, false otherwise
   * @throws IOException communication failure
   */
  boolean componentExists(ComponentId componentId) throws IOException;

  /**
   * Deploys a policy at the local PDP of the enforcement scope.
   *
   * @param policyId the id of the policy to be deployed. Will be retrieved via
   *          the PRP
   * @return true, if the policy was deployed successfully, false otherwise
   * @throws IOException if connection problem occurs
   * @throws NoSuchElementException policy not found in PMP database
   * @throws ConflictingPolicyException multiple policies with same policyId
   *           exist in PMP database
   */
  boolean deployPolicy(PolicyId policyId) throws IOException, NoSuchElementException, ConflictingPolicyException;

  /**
   * Gets all policy.
   *
   * @param scope the scope
   * @return all policy
   * @throws IOException Signals that an I/O exception has occurred.
   */
  List<Policy> getDeployedPolicies(EnforcementScopeId scope) throws IOException;

  /**
   * Gets the status.
   *
   * @param policyId the policy id
   * @return the status
   * @throws IOException Signals that an I/O exception has occurred.
   */
  boolean getDeploymentStatus(PolicyId policyId) throws IOException;

  /**
   * Describes the enforcement scope as a set of registered components and
   * corresponding interfaces.
   *
   * @param scope the scope
   * @param includeInterface a flag that indicates whether the interface
   *          description of the component should be included in the result
   * @return an {@link EnforcementScope} of the local enforcement scope
   * @throws IOException if connection problem occurs
   */
  EnforcementScope getEnforcementScope(EnforcementScopeId scope, boolean includeInterface) throws IOException;

  /**
   * Gets all policy.
   *
   * @param scope the scope
   * @return all policy
   * @throws IOException Signals that an I/O exception has occurred.
   */
  List<Policy> getPolicies(EnforcementScopeId scope) throws IOException;

  /**
   * Gets the latest policy.
   *
   * @param policyId the policy id
   * @return the latest policy
   * @throws IOException Signals that an I/O exception has occurred.
   */
  Policy getPolicy(PolicyId policyId) throws IOException;

  /**
   * Gets all policy ids.
   *
   * @param scope the scope
   * @return all policy IDs
   * @throws IOException Signals that an I/O exception has occurred.
   */
  List<PolicyId> getPolicyIds(EnforcementScopeId scope) throws IOException;

  /**
   * Provides a list of currently deployed policy ids.
   *
   * @param scope the scope
   * @return list of currently deployed policy ids
   * @throws IOException communication failure
   */
  List<PolicyId> listDeployedPolicies(EnforcementScopeId scope) throws IOException;

  /**
   * Lookup by {@link ComponentId}.
   *
   * @param id the id
   * @return the component
   * @throws IOException communication failure
   */
  ComponentBase lookup(ComponentId id) throws IOException;

  /**
   * Returns the list of PDPs in the database.
   *
   * @return the PDP in the database
   * @throws IOException communication failure
   */
  Component lookupPdp() throws IOException;

  /**
   * Lookup by {@link ComponentId}.
   *
   * @param id the id
   * @return the component
   * @throws IOException communication failure
   */
  PepComponent lookupPep(ComponentId id) throws IOException;

  /**
   * Returns the list of PEPs in the database.
   *
   * @param scope the scope
   * @param includeInterface flag to include the interface (URLs and
   *          {@link MethodInterfaceDescription}s
   * @return the list of PEPs in the database
   * @throws IOException communication failure
   */
  List<PepComponent> lookupPep(EnforcementScopeId scope, boolean includeInterface) throws IOException;

  /**
   * Returns the list of PIPs in the database.
   *
   * @param scope the scope
   * @param includeInterface flag to include the interface (URLs and
   *          {@link MethodInterfaceDescription}s
   * @return the list of PIPs in the database
   * @throws IOException communication failure
   */
  List<Component> lookupPip(EnforcementScopeId scope, boolean includeInterface) throws IOException;

  /**
   * Returns the list of PIPs in the database.
   *
   * @param scope the scope
   * @param includeInterface flag to include the interface (URLs and
   *          {@link MethodInterfaceDescription}s
   * @param query the required / available {@link MethodInterfaceDescription}s
   *          to match against
   * @return the list of PIPs in the database
   * @throws IOException communication failure
   */
  List<Component> lookupPip(EnforcementScopeId scope, boolean includeInterface, MethodInterfaceDescription query) throws IOException;

  /**
   * Returns the list of PXPs in the database.
   *
   * @param scope the scope
   * @param includeInterface flag to include the interface (URLs and
   *          {@link MethodInterfaceDescription}s
   * @return the list of PXPs in the database
   * @throws IOException communication failure
   */
  List<Component> lookupPxp(EnforcementScopeId scope, boolean includeInterface) throws IOException;

  /**
   * Returns the list of PXPs in the database.
   *
   * @param scope the scope
   * @param includeInterface flag to include the interface (URLs and
   *          {@link MethodInterfaceDescription}s
   * @param query the required / available {@link MethodInterfaceDescription}s
   *          to match against
   * @return the list of PXPs in the database
   * @throws IOException communication failure
   */
  List<Component> lookupPxp(EnforcementScopeId scope, boolean includeInterface, MethodInterfaceDescription query) throws IOException;

  /**
   * Adds a component into the database.
   *
   * @param component the component
   * @return true if the component was successfully added, false otherwise
   * @throws IOException communication failure
   */
  boolean registerComponent(Component component) throws IOException;

  /**
   * Adds a pep into the database.
   *
   * @param component the component
   * @return true if the component was successfully added, false otherwise
   * @throws IOException communication failure
   */
  boolean registerPEPComponent(PepComponent component) throws IOException;

  /**
   * Removes the policy.
   *
   * @param policy the policy
   * @param reason the reason
   * @return true, if successful
   * @throws IOException Signals that an I/O exception has occurred.
   */
  boolean removePolicy(PolicyId policy, String reason) throws IOException;

  /**
   * Revokes a currently deployed policy.
   *
   * @param policyId the ID of the policy to be revoked
   * @return true, if the policy was revoked successfully, false otherwise
   * @throws IOException if connection problem occurs
   * @throws PolicyRevokationException remove is not successful
   */
  boolean revokePolicy(PolicyId policyId) throws IOException, PolicyRevokationException;

  /**
   * Sets the status.
   *
   * @param policyId the policy id
   * @param deployed the deployed
   * @throws IOException Signals that an I/O exception has occurred.
   */
  void setDeploymentStatus(PolicyId policyId, boolean deployed) throws IOException;

  /**
   * Deletes a component from the database.
   *
   * @param componentId the ID of the component to be removed
   * @return true if removed successfully (or did not exist), false otherwise
   * @throws IOException communication failure
   */
  boolean unregisterComponent(ComponentId componentId) throws IOException;

  /**
   * Increases the counter of processed events
   *
   * @param scope the scope
   * @param increment the number to add
   * @return true, if successful, false otherwise
   * @throws IOException
   */
  boolean incrementRequestCounter(EnforcementScopeId scope, long increment) throws IOException;

  /**
   * Returns the list of solutions that is currently blacklisted
   *
   * @return the list of solutions that is currently blacklisted
   * @throws IOException
   */
  List<EnforcementScopeId> getBlackListedSolutions() throws IOException;

  /**
   * Gets the running hashcode of the pmp. Used in development mode, to check if
   * the pmp restarted and the pdp has to be reset.
   *
   * @return the running id
   * @throws IOException Signals that an I/O exception has occurred.
   */
  Integer getPmpHashcode() throws IOException;
}
