/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.component.interfaces;

import de.fraunhofer.iese.ind2uce.api.component.exception.ConflictingPolicyException;
import de.fraunhofer.iese.ind2uce.api.component.exception.EvaluationUndecidableException;
import de.fraunhofer.iese.ind2uce.api.component.exception.PolicyRevokationException;
import de.fraunhofer.iese.ind2uce.api.component.identifier.EnforcementScopeId;
import de.fraunhofer.iese.ind2uce.api.policy.AuthorizationDecision;
import de.fraunhofer.iese.ind2uce.api.policy.Event;
import de.fraunhofer.iese.ind2uce.api.policy.identifier.PolicyId;

import java.io.IOException;
import java.util.List;
import java.util.Map;

/**
 * The PolicyDecisionPoint draws decisions based on the currently deployed
 * policies.
 *
 * @author Fraunhofer IESE
 */
public interface IPolicyDecisionPoint extends IComponent {

  /**
   * Clears all caches of the PDP (response and connector caches).
   *
   * @return true, if cache was cleaned
   * @throws IOException if there occurs a connection problem.
   */
  boolean clearAllCaches() throws IOException;

  /**
   * Notifies the PDP about a certain event. Based on the events, the policies
   * are evaluated and decision is drawn.
   *
   * @param event the occurred event
   * @return the {@link AuthorizationDecision} with respect to the event and
   *         deployed policies
   * @throws IOException if there occurs a connection problem.
   * @throws IllegalArgumentException if event is semantically invalid
   * @throws EvaluationUndecidableException if PDP can't evaluate the event due
   *           to internal ambiguity.
   */
  AuthorizationDecision decisionRequest(Event event) throws IOException, IllegalArgumentException, EvaluationUndecidableException;

  /**
   * Deploys a policy directly.
   *
   * @param policy the policy in XML format
   * @return true if the policy is deployed, false otherwise
   * @throws IOException if there occurs a connection problem.
   * @throws IllegalArgumentException if policyId is syntactically invalid
   * @throws ConflictingPolicyException if multiple policy with same id are
   *           found in PMP
   */
  boolean deploy(String policy) throws IOException, IllegalArgumentException, ConflictingPolicyException;

  /**
   * Provides a list of currently deployed policy names.
   *
   * @return list of currently deployed policy names
   * @throws IOException if there occurs a connection problem.
   */
  List<PolicyId> listDeployedPolicies() throws IOException;

  /**
   * Revokes a policy.
   *
   * @param policyId the name of the policy to be revoked
   * @return true if the policy is revoked, false otherwise
   * @throws IOException if there occurs a connection problem.
   * @throws IllegalArgumentException if policyId is syntactically invalid
   * @throws PolicyRevokationException fails to revoke the policy
   */
  boolean revokePolicy(PolicyId policyId) throws IOException, IllegalArgumentException, PolicyRevokationException;

  boolean addToBlacklist(List<EnforcementScopeId> ids) throws IOException;

  boolean removeFromBlacklist(List<EnforcementScopeId> ids) throws IOException;

  Map<Event, AuthorizationDecision> timerDecisionRequest(List<Event> events) throws IOException, EvaluationUndecidableException;

}
