/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.component.identifier;

import de.fraunhofer.iese.ind2uce.api.common.Ind2uceEntity;

import java.util.regex.Pattern;

/**
 * The Class EnforcementScopeId.
 */
public class EnforcementScopeId extends Ind2uceEntity {

  /**
   * The Constant serialVersionUID.
   */
  private static final long serialVersionUID = 5971783346726612578L;

  /***
   * pattern for enforcement scope ^urn:es:[a-z0-9()+,\-.=@;$_!*']+$
   */
  public static final Pattern URN_ES_PATTERN = Pattern.compile("^urn:es:[a-z0-9()+,\\-.=@;$_!*']+$", Pattern.CASE_INSENSITIVE);

  /**
   * The identifier.
   */
  private final String identifier;

  /**
   * Instantiates a new enforcement scope id.
   *
   * @param urn the urn
   */
  public EnforcementScopeId(String urn) {

    if (urn == null) {
      throw new IllegalArgumentException("Illegal Enforcement Scope ID");
    } else if (!urn.startsWith("urn:es:")) {
      urn = "urn:es:" + urn;
    }

    this.validate(urn);
    if (URN_ES_PATTERN.matcher(urn).find()) {
      final String[] componentIdParts = urn.split(":");
      this.identifier = componentIdParts[2];
    } else {
      throw new IllegalArgumentException(urn);
    }
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (!(o instanceof EnforcementScopeId)) {
      return false;
    }

    final EnforcementScopeId that = (EnforcementScopeId)o;

    return this.identifier != null ? this.identifier.equals(that.identifier) : that.identifier == null;

  }

  /**
   * Gets the identifier.
   *
   * @return the identifier
   */
  public String getIdentifier() {
    return this.identifier;
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#hashCode()
   */
  @Override
  public int hashCode() {
    return this.identifier != null ? this.identifier.hashCode() : 0;
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.common.Ind2uceEntity#toString()
   */
  @Override
  public String toString() {
    final StringBuilder s = new StringBuilder();
    s.append("urn:es:");
    s.append(this.identifier);
    return s.toString();
  }

  /**
   * Validate.
   *
   * @param uri the uri
   */
  protected void validate(String uri) {
    URN_ES_PATTERN.matcher(uri).matches();
  }
}
