/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.component.identifier;

import de.fraunhofer.iese.ind2uce.api.common.Ind2uceEntity;
import de.fraunhofer.iese.ind2uce.api.component.ComponentType;

import java.util.UUID;
import java.util.regex.Pattern;

import javax.persistence.Column;
import javax.persistence.Embeddable;
import javax.persistence.EnumType;
import javax.persistence.Enumerated;

/**
 * A component ID uniquely names an IND&sup2;UCE component. The ID is an URN
 * which must follow the format
 * &quot;urn:component:&lt;scope&gt;:&lt;p*p&gt;:&lt;name&gt;&quot;, where
 * &quot;&lt;p*p&gt;&quot; can be pap, pdp, pep, pip, pmp, pmp_server, prp, or
 * pxp.
 */
@Embeddable
public class ComponentId extends Ind2uceEntity {

  /**
   * The Constant serialVersionUID.
   */
  private static final long serialVersionUID = 3233135419998235135L;

  /**
   * The pattern of a component ID.
   * ^urn:component:([a-z0-9()+,\\-.=@;$_!*'])+:(p[adeimrx]p|pmp_server):([a-z0-9()+,\\-.=@;$_!*']+|%[0-9a-f]{2})$
   */
  public static final Pattern URN_COMPONENT_PATTERN = Pattern.compile("^urn:component:([a-z0-9()+,\\-.=@;$_!*'])+:(p[adeimrx]p|pmp_server):([a-z0-9()+,\\-.=@;$_!*']+|%[0-9a-f]{2})$",
      Pattern.CASE_INSENSITIVE);

  /**
   * The component type.
   */
  @Column(name = "component_type", length = 20)
  @Enumerated(EnumType.STRING)
  private ComponentType componentType;

  /**
   * The identifier.
   */
  @Column(name = "identifier", length = 70)
  private String identifier;

  /**
   * The scope.
   */
  @Column(name = "scope", length = 70)
  private String scope;

  /**
   * Parameterless constructor is required for JPA.
   */
  public ComponentId() {
    super();
  }

  /**
   * Generates a random ID for the given component type.
   *
   * @param type The component type for which the ID is intended.
   */
  public ComponentId(ComponentType type) {
    this(("urn:component:scope:" + type.toString() + ":" + UUID.randomUUID().toString()).toLowerCase());
  }

  /**
   * Instantiates a new component id.
   *
   * @param id the id
   */
  public ComponentId(String id) {
    this.validate(id);
    this.setAttributesByUrn(id, this);
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (!(o instanceof ComponentId)) {
      return false;
    }

    final ComponentId that = (ComponentId)o;

    if (this.componentType != that.componentType) {
      return false;
    }
    return (this.identifier != null ? this.identifier.equals(that.identifier) : that.identifier == null) && (this.scope != null ? this.scope.equals(that.scope) : that.scope == null);

  }

  /**
   * Gets the component type.
   *
   * @return the component type
   */
  public ComponentType getComponentType() {
    return this.componentType;
  }

  /**
   * Gets the id.
   *
   * @return the id
   */
  public String getId() {
    return this.toString();
  }

  /**
   * Gets the identifier.
   *
   * @return the identifier
   */
  public String getIdentifier() {
    return this.identifier;
  }

  /**
   * Gets the scope.
   *
   * @return the scope
   */
  public String getScope() {
    return this.scope;
  }

  /**
   * Gets the type.
   *
   * @return the type
   */
  public ComponentType getType() {
    return this.componentType;
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#hashCode()
   */
  @Override
  public int hashCode() {
    int result = super.hashCode();
    result = 31 * result + (this.scope != null ? this.scope.hashCode() : 0);
    result = 31 * result + (this.componentType != null ? this.componentType.hashCode() : 0);
    result = 31 * result + (this.identifier != null ? this.identifier.hashCode() : 0);
    return result;
  }

  /**
   * Extracts the component's attributes from the urn and sets them to the given
   * component-instance.
   *
   * @param urn the urn
   * @param instance the instance
   */
  private void setAttributesByUrn(String urn, ComponentId instance) {
    if (URN_COMPONENT_PATTERN.matcher(urn).find()) {
      final String[] componentIdParts = urn.split(":");
      if (!componentIdParts[0].equalsIgnoreCase("urn")) {
        throw new IllegalArgumentException(urn);
      }
      if (!componentIdParts[1].equalsIgnoreCase("component")) {
        throw new IllegalArgumentException(urn);
      }
      instance.setScope(componentIdParts[2]);
      instance.setComponentType(ComponentType.valueOf(componentIdParts[3].toUpperCase()));
      instance.setIdentifier(componentIdParts[4]);
    } else {
      throw new IllegalArgumentException(urn);
    }
  }

  /**
   * Sets the component type.
   *
   * @param componentType the new component type
   */
  public void setComponentType(ComponentType componentType) {
    this.componentType = componentType;
  }

  /**
   * Sets the identifier.
   *
   * @param identifier the new identifier
   */
  public void setIdentifier(String identifier) {
    this.identifier = identifier;
  }

  /**
   * Sets the scope.
   *
   * @param scope the new scope
   */
  public void setScope(String scope) {
    this.scope = scope;
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.api.common.Ind2uceEntity#toString()
   */
  @Override
  public String toString() {
    final StringBuilder s = new StringBuilder();
    s.append("urn:component:");
    s.append(this.getScope());
    s.append(":");
    s.append(this.getType().name().toLowerCase());
    s.append(":");
    s.append(this.getIdentifier());
    return s.toString();
  }

  /**
   * Validates a string with respect to the URN_COMPONENT_PATTERN.
   *
   * @param uri the uri
   */
  protected void validate(String uri) {
    URN_COMPONENT_PATTERN.matcher(uri).matches();
  }
}
