/*
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.component.description;

import de.fraunhofer.iese.ind2uce.api.common.Ind2uceEntity;
import de.fraunhofer.iese.ind2uce.api.policy.identifier.ActionId;

import org.hibernate.annotations.Fetch;
import org.hibernate.annotations.FetchMode;

import java.util.List;

import javax.persistence.CascadeType;
import javax.persistence.Embedded;
import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.JoinColumn;
import javax.persistence.OneToMany;

/**
 * Describes the capabilities of Policy Enforcement Points in terms of
 * intercepted events and enforceable methods.
 *
 * @author Fraunhofer IESE
 */
@Entity
public class PepInterfaceDescription extends Ind2uceEntity {

  /**
   * The id.
   */
  @Id
  @GeneratedValue(strategy = GenerationType.IDENTITY)
  protected Long id;

  /**
   * A description of the method.
   */
  protected String description;

  /**
   * The event that is intercepted or monitored.
   */
  @Embedded
  private ActionId event;

  /**
   * A flag that indicated whether the event can be modified or blocked.
   */
  private boolean isPreventive;

  /** * List of parameter. */
  @OneToMany(cascade = CascadeType.ALL, orphanRemoval = true)
  @JoinColumn(name = "interface_event_parameter_description_id")
  @Fetch(FetchMode.SUBSELECT)
  private List<InputParameterDescription> eventParameterDescription;

  /**
   * Default Constructor, at least required by JPA.s
   */
  public PepInterfaceDescription() {

  }

  /**
   * Instantiates a new pep interface description.
   *
   * @param event the event that the PEP intercepts or monitors
   * @param isPreventive a flag that indicated whether the event can be modified
   *          or blocked
   * @param description the description
   */
  public PepInterfaceDescription(ActionId event, boolean isPreventive, String description) {
    this(event, isPreventive, description, null);
  }

  /**
   * Instantiates a new pep interface description.
   *
   * @param event the event that the PEP intercepts or monitors
   * @param isPreventive a flag that indicated whether the event can be modified
   *          or blocked
   * @param description the description
   * @param eventInputParameters the eventInputParameters
   */
  public PepInterfaceDescription(ActionId event, boolean isPreventive, String description, List<InputParameterDescription> eventInputParameters) {
    this.description = description;
    this.event = event;
    this.isPreventive = isPreventive;
    this.eventParameterDescription = eventInputParameters;

  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (!(o instanceof PepInterfaceDescription)) {
      return false;
    }

    final PepInterfaceDescription that = (PepInterfaceDescription)o;

    if (this.isPreventive != that.isPreventive) {
      return false;
    }
    if (this.id != null ? !this.id.equals(that.id) : that.id != null) {
      return false;
    }
    if (this.description != null ? !this.description.equals(that.description) : that.description != null) {
      return false;
    }
    if (this.event != null ? !this.event.equals(that.event) : that.event != null) {
      return false;
    }
    return this.eventParameterDescription != null ? this.eventParameterDescription.equals(that.eventParameterDescription) : that.eventParameterDescription == null;

  }

  /**
   * Gets the description.
   *
   * @return the description
   */
  public String getDescription() {
    return this.description;
  }

  /**
   * Gets the event that is intercepted or monitored.
   *
   * @return the event that is intercepted or monitored
   */
  public ActionId getEvent() {
    return this.event;
  }

  /**
   * Gets the event parameter description.
   *
   * @return the event parameter description
   */
  public List<InputParameterDescription> getEventParameterDescription() {
    return this.eventParameterDescription;
  }

  /**
   * Gets the id.
   *
   * @return the id
   */
  public Long getId() {
    return this.id;
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#hashCode()
   */
  @Override
  public int hashCode() {
    int result = this.id != null ? this.id.hashCode() : 0;
    result = 31 * result + (this.description != null ? this.description.hashCode() : 0);
    result = 31 * result + (this.event != null ? this.event.hashCode() : 0);
    result = 31 * result + (this.isPreventive ? 1 : 0);
    result = 31 * result + (this.eventParameterDescription != null ? this.eventParameterDescription.hashCode() : 0);
    return result;
  }

  /**
   * Checks if is a flag that indicated whether the event can be modified or
   * blocked.
   *
   * @return the a flag that indicated whether the event can be modified or
   *         blocked
   */
  public boolean isPreventive() {
    return this.isPreventive;
  }

  /**
   * Sets the description.
   *
   * @param description the new description
   */
  public void setDescription(String description) {

    this.description = description;
  }

  /**
   * Sets the event that is intercepted or monitored.
   *
   * @param event the new event that is intercepted or monitored
   */
  public void setEvent(ActionId event) {
    this.event = event;
  }

  /**
   * Sets the event parameter description.
   *
   * @param eventParameterDescription the new event parameter description
   */
  public void setEventParameterDescription(List<InputParameterDescription> eventParameterDescription) {
    this.eventParameterDescription = eventParameterDescription;
  }

  /**
   * Sets the id.
   *
   * @param id the new id
   */
  public void setId(Long id) {
    this.id = id;
  }

  /**
   * Sets the a flag that indicated whether the event can be modified or
   * blocked.
   *
   * @param preventive the new a flag that indicated whether the event can be
   *          modified or blocked
   */
  public void setPreventive(boolean preventive) {
    this.isPreventive = preventive;
  }

}
