/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.component.description;

import de.fraunhofer.iese.ind2uce.api.common.Ind2uceEntity;

import com.google.common.base.Objects;

import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.hibernate.annotations.Fetch;
import org.hibernate.annotations.FetchMode;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

import javax.persistence.CascadeType;
import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.JoinColumn;
import javax.persistence.OneToMany;

/**
 * Describes an interface of a method. Contains of a name, a list of parameters
 * and a return type.
 *
 * @author Fraunhofer IESE
 */
@Entity
public class MethodInterfaceDescription extends Ind2uceEntity {

  /**
   * the Constant serialVersionUID.
   */
  private static final long serialVersionUID = 6925352815906821374L;

  /**
   * The id.
   */
  @Id
  @GeneratedValue(strategy = GenerationType.IDENTITY)
  protected Long id;

  /**
   * A description of the method.
   */
  protected String description;

  /**
   * The name of the method.
   */
  protected String methodName;

  /**
   * The return type of the method.
   */
  protected String returnType;

  /**
   * A list of parameters.
   */
  @OneToMany(cascade = CascadeType.ALL, orphanRemoval = true)
  @JoinColumn(name = "interface_description_id")
  @Fetch(FetchMode.SUBSELECT)
  private List<InputParameterDescription> parameters = new ArrayList<>();

  /**
   * Instantiates a new method interface description.
   */
  MethodInterfaceDescription() {

  }

  /**
   * Instantiates a new interface description.
   *
   * @param methodName name of the method
   * @param returnType return type of the method
   * @param description description of the method
   */
  public MethodInterfaceDescription(String methodName, Class<?> returnType, String description) {
    this(methodName, returnType, description, new ArrayList<InputParameterDescription>());
  }

  /**
   * Instantiates a new interface description.
   *
   * @param methodName name of the method
   * @param returnType return type of the method
   * @param description description of the method
   * @param parameters a list of parameters for the method
   */
  public MethodInterfaceDescription(String methodName, Class<?> returnType, String description, InputParameterDescription... parameters) {
    this(methodName, returnType, description, new LinkedList<>(Arrays.asList(parameters)));
  }

  /**
   * Instantiates a new interface description.
   *
   * @param methodName name of the method
   * @param returnType return type of the method
   * @param description description of the method
   * @param parameters a list of parameters for the method
   */
  public MethodInterfaceDescription(String methodName, Class<?> returnType, String description, List<InputParameterDescription> parameters) {
    this.description = description;
    this.methodName = methodName;
    this.description = description;

    if (parameters != null) {
      this.parameters = parameters;
    } else {
      this.parameters.clear();
    }

    this.setReturnType(returnType);
  }

  /**
   * Adds the parameter.
   *
   * @param param the param
   */
  public void addParameter(InputParameterDescription param) {
    this.parameters.add(param);
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  public boolean equals(Object obj) {
    if (obj instanceof MethodInterfaceDescription) {
      final MethodInterfaceDescription description = (MethodInterfaceDescription)obj;
      if (!Objects.equal(this.getReturnType(), description.getReturnType())) {
        return false;
      }
      if (!Objects.equal(this.getMethodName(), description.getMethodName())) {
        return false;
      }
      if (!Objects.equal(this.getDescription(), description.getDescription())) {
        return false;
      }
      return Objects.equal(this.parameters, description.getParameters());
    }
    return false;
  }
  /*
   * (non-Javadoc)
   * @see java.lang.Object#hashCode()
   */

  /**
   * Gets the a description of the method.
   *
   * @return the a description of the method
   */
  public String getDescription() {
    return this.description;
  }

  /**
   * Gets the id.
   *
   * @return the id
   */
  public Long getId() {
    return this.id;
  }

  /**
   * Gets the name of the method.
   *
   * @return the name of the method
   */
  public String getMethodName() {
    return this.methodName;
  }

  /**
   * Gets the a list of parameters.
   *
   * @return the a list of parameters
   */
  public List<InputParameterDescription> getParameters() {
    return this.parameters;
  }

  /**
   * Gets the return type of the method.
   *
   * @return the return type of the method
   */
  public Class<?> getReturnType() {
    return TypeByName.getClassForName(this.returnType);
  }

  /**
   * Gets the return type name.
   *
   * @return the return type name
   */
  public String getReturnTypeName() {
    return this.returnType;
  }

  @Override
  public int hashCode() {
    final HashCodeBuilder builder = new HashCodeBuilder(17, 31);
    builder.append(this.id);
    builder.append(this.methodName);
    builder.append(this.returnType);
    builder.append(this.parameters);
    builder.append(this.description);
    return builder.toHashCode();
  }

  /**
   * Sets the a description of the method.
   *
   * @param description the new a description of the method
   */
  public void setDescription(String description) {
    this.description = description;
  }

  /**
   * Sets the id.
   *
   * @param id the new id
   */
  public void setId(Long id) {
    this.id = id;
  }

  /**
   * Sets the name of the method.
   *
   * @param methodName the new name of the method
   */
  public void setMethodName(String methodName) {
    this.methodName = methodName;
  }

  /**
   * Sets the a list of parameters.
   *
   * @param parameters the new a list of parameters
   */
  public void setParameters(List<InputParameterDescription> parameters) {
    this.parameters = parameters;
  }

  /**
   * Sets the return type of the method.
   *
   * @param returnType the new return type of the method
   */
  public void setReturnType(Class<?> returnType) {
    if (returnType != null) {
      Class<?> wrappedType = Primitives.replaceByWrapper(returnType);
      if (wrappedType != null && Number.class.isAssignableFrom(wrappedType)) {
        wrappedType = Number.class;
        this.returnType = wrappedType.getCanonicalName();
      }
      this.returnType = returnType.getCanonicalName();
    }
  }

}
