/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.component.description;

import de.fraunhofer.iese.ind2uce.api.common.Ind2uceEntity;

import org.apache.commons.lang3.builder.HashCodeBuilder;

import java.lang.reflect.Type;
import java.util.List;

import javax.persistence.CascadeType;
import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.OneToMany;

/**
 * The Class InputParameterDescription is used by different component as PXP,
 * PIP, PEP and others to specify {@link MethodInterfaceDescription} at the time
 * of registration at PMP.
 */
@Entity
public class InputParameterDescription extends Ind2uceEntity {

  /**
   * The Constant serialVersionUID.
   */
  private static final long serialVersionUID = -5113596900510393171L;

  /**
   * The id.
   */
  @Id
  @GeneratedValue(strategy = GenerationType.IDENTITY)
  private Long id;

  /**
   * The name of the parameter.
   */
  private String name;

  /**
   * A description about what the parameter is used for.
   */
  private String description;

  /**
   * A regular expression, the parameter has to adhere to.
   */
  private String pattern;

  /**
   * The parameter type.
   */
  private String type;

  /**
   * The Json type of the parameter like ARRAY, OBJECT or PRIMITIVE.
   */
  private JsonType jsonType;

  /**
   * A flag, indicating whether the parameter is mandatory or not.
   */
  boolean mandatory = true;

  /**
   * string representing the type description
   */
  private String typeDescription;

  /** The parameter type description. */
  @OneToMany(cascade = {
      CascadeType.ALL
  }, orphanRemoval = true)
  private List<TypeDescription> parameterTypeDescription;

  /**
   * Instantiates a new input parameter description for JPA.
   */
  InputParameterDescription() {

  }

  /**
   * Instantiates a new input parameter description.
   *
   * @param name the name of the parameter
   * @param description the description of the parameter
   * @param type the type of the parameter
   */
  public InputParameterDescription(String name, String description, Class<?> type) {
    this(name, description, null, true, type);
  }

  /**
   * Instantiates a new input parameter description.
   *
   * @param name the name of the parameter
   * @param description the description of the parameter
   * @param pattern the pattern the parameter value has to adhere to
   * @param mandatory indication a parameter that is mandatory
   * @param type the type of the parameter
   */
  public InputParameterDescription(String name, String description, String pattern, boolean mandatory, Class<?> type) {
    this.name = name;
    this.description = description;
    this.pattern = pattern;
    this.mandatory = mandatory;
    this.setType(type);
  }

  /**
   * Instantiates a new input parameter description.
   *
   * @param name the name
   * @param description the description
   * @param pattern the pattern
   * @param mandatory the mandatory
   * @param type the type
   */
  public InputParameterDescription(String name, String description, String pattern, boolean mandatory, Type type) {
    this.name = name;
    this.description = description;
    this.pattern = pattern;
    this.mandatory = mandatory;
    this.type = type.toString();
  }

  /**
   * Instantiates a new input parameter description.
   *
   * @param name the name
   * @param description the description
   * @param pattern the pattern
   * @param mandatory the mandatory
   * @param type the type
   * @param jsonType the json type
   * @param parameterTypeDescription the parameter type description
   */
  public InputParameterDescription(String name, String description, String pattern, boolean mandatory, Type type, JsonType jsonType, List<TypeDescription> parameterTypeDescription) {
    this(name, description, pattern, mandatory, type);
    this.setJsonType(jsonType);
    this.parameterTypeDescription = parameterTypeDescription;
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  public boolean equals(Object obj) {
    if (obj == null || !(obj instanceof InputParameterDescription)) {
      return false;
    }
    final InputParameterDescription other = (InputParameterDescription)obj;
    return this.name.equals(other.getName()) && this.type.equals(other.getTypeName());
  }

  /**
   * Gets the a description about what the parameter is used for.
   *
   * @return the a description about what the parameter is used for
   */
  public String getDescription() {
    return this.description;
  }

  /**
   * Gets the id.
   *
   * @return the id
   */
  public Long getId() {
    return this.id;
  }

  /**
   * @return the jsonType
   */
  public JsonType getJsonType() {
    return this.jsonType;
  }

  /**
   * Gets the name.
   *
   * @return the name
   */
  public String getName() {
    return this.name;
  }

  /**
   * Gets the parameter type description.
   *
   * @return the parameter type description
   */
  public List<TypeDescription> getParameterTypeDescription() {
    return this.parameterTypeDescription;
  }

  /**
   * Gets the a regular expression, the parameter has to adhere to.
   *
   * @return the a regular expression, the parameter has to adhere to
   */
  public String getPattern() {
    return this.pattern;
  }

  /**
   * Gets the parameter type.
   *
   * @return the parameter type
   */
  public Class<?> getType() {
    return TypeByName.getClassForName(this.type);
  }

  /**
   * Gets the type description.
   *
   * @return the type description
   */
  public String getTypeDescription() {
    return this.typeDescription;
  }

  /**
   * Gets the type name.
   *
   * @return the type name
   */
  public String getTypeName() {
    return this.type;
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#hashCode()
   */
  @Override
  public int hashCode() {
    final HashCodeBuilder builder = new HashCodeBuilder(17, 31);
    builder.append(this.name);
    builder.append(this.type);
    return builder.toHashCode();
  }

  /**
   * Checks if is a flag, indicating whether the parameter is mandatory or not.
   *
   * @return the a flag, indicating whether the parameter is mandatory or not
   */
  public boolean isMandatory() {
    return this.mandatory;
  }

  /**
   * Sets the a description about what the parameter is used for.
   *
   * @param description the new a description about what the parameter is used
   *          for
   */
  public void setDescription(String description) {
    this.description = description;
  }

  /**
   * Sets the id.
   *
   * @param id the new id
   */
  public void setId(Long id) {
    this.id = id;
  }

  /**
   * @param jsonType the jsonType to set
   */
  public void setJsonType(JsonType jsonType) {
    this.jsonType = jsonType;
  }

  /**
   * Sets the a flag, indicating whether the parameter is mandatory or not.
   *
   * @param mandatory the new a flag, indicating whether the parameter is
   *          mandatory or not
   */
  public void setMandatory(boolean mandatory) {
    this.mandatory = mandatory;
  }

  /**
   * Sets the name.
   *
   * @param name the new name
   */
  public void setName(String name) {
    this.name = name;
  }

  /**
   * Sets the parameter type description.
   *
   * @param parameterTypeDescription the new parameter type description
   */
  public void setParameterTypeDescription(List<TypeDescription> parameterTypeDescription) {
    this.parameterTypeDescription = parameterTypeDescription;
  }

  /**
   * Sets the a regular expression, the parameter has to adhere to.
   *
   * @param pattern the new a regular expression, the parameter has to adhere to
   */
  public void setPattern(String pattern) {
    this.pattern = pattern;
  }

  /**
   * Sets the parameter type.
   *
   * @param type the new parameter type
   */
  public void setType(Class<?> type) {
    Class<?> wrappedType = Primitives.replaceByWrapper(type);
    if(wrappedType != null && Number.class.isAssignableFrom(wrappedType))
      wrappedType = Number.class;
    this.type = wrappedType.getCanonicalName();
  }

  /**
   * Sets the type description.
   *
   * @param typeDescription the new type description
   */
  public void setTypeDescription(String typeDescription) {
    this.typeDescription = typeDescription;
  }
}
