/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.component;

import de.fraunhofer.iese.ind2uce.api.common.Ind2uceEntity;
import de.fraunhofer.iese.ind2uce.api.component.identifier.ComponentId;

import java.net.URI;
import java.util.ArrayList;
import java.util.List;

import javax.persistence.CollectionTable;
import javax.persistence.Column;
import javax.persistence.ElementCollection;
import javax.persistence.EmbeddedId;
import javax.persistence.Entity;
import javax.persistence.Inheritance;
import javax.persistence.InheritanceType;
import javax.persistence.JoinColumn;
import javax.persistence.Lob;

/**
 * Created by eitel on 12/1/16.
 */
@Entity
@Inheritance(strategy = InheritanceType.TABLE_PER_CLASS)
public abstract class ComponentBase extends Ind2uceEntity {
  /**
   * The unique ID of the {@link Component}.
   */
  @EmbeddedId
  protected ComponentId id;

  /**
   * A list of {@link URI}s that can be used to communicate with the
   * {@link Component}.
   */
  @Lob
  @ElementCollection
  @CollectionTable(name = "component_urls", joinColumns = {
      @JoinColumn(referencedColumnName = "component_type", name = "component_type"), @JoinColumn(referencedColumnName = "identifier", name = "identifier"),
      @JoinColumn(referencedColumnName = "scope", name = "scope")
  })
  @Column(name = "url")
  protected List<URI> urls;

  public ComponentBase() {

  }

  public ComponentBase(final ComponentId cId) {
    this(cId, new ArrayList<>());
  }

  public ComponentBase(final ComponentId cId, final List<URI> urls) {
    if (cId == null) {
      throw new IllegalArgumentException("ComponentId may not be null");
    }
    this.id = cId;
    this.urls = urls;
  }

  /**
   * Adds a URL to the list of {@link URI}s that can be used to communicate with
   * the {@link Component}.
   *
   * @param url the url to add
   */
  public void addUrl(URI url) {
    if (null == url) {
      return;
    }
    if (this.urls == null) {
      this.urls = new ArrayList<>();
    }
    this.urls.add(url);
  }

  /**
   * Gets the unique ID of the {@link Component}.
   *
   * @return the unique ID of the {@link Component}
   */
  public ComponentId getId() {
    return this.id;
  }

  /**
   * Returns the {@link ComponentType} of the component.
   *
   * @return the {@link ComponentType} of the component
   */
  public ComponentType getType() {
    return this.id.getType();
  }

  /**
   * Gets the a list of {@link URI}s that can be used to communicate with the
   * {@link Component}.
   *
   * @return a list of {@link URI}s that can be used to communicate with the
   *         {@link Component}.
   */
  public List<URI> getUrls() {
    return this.urls;
  }

  /**
   * Provides a list of URLs of a certain protocol/scheme.
   *
   * @param protocols the list of protocols to be used linked by an
   *          &quot;or&quot; function
   * @return a list of {@link URI}s that can be used to communicate with the
   *         {@link Component}
   */
  public List<URI> getUrlsForProtocol(String... protocols) {
    if (protocols.length == 0) {
      return this.getUrls();
    }
    final List<URI> result = new ArrayList<>();
    for (final URI url : this.urls) {
      for (final String protocol : protocols) {
        if (protocol.equals(url.getScheme())) {
          result.add(url);
          break;
        }
      }
    }
    return result;
  }

  /**
   * Sets the unique ID of the {@link Component}.
   *
   * @param id the new unique ID of the {@link Component}
   */
  public void setId(ComponentId id) {
    this.id = id;
  }

  /**
   * Sets the a list of {@link URI}s that can be used to communicate with the
   * {@link Component}.
   *
   * @param urls a list of {@link URI}s that can be used to communicate with the
   *          {@link Component}.
   */
  public void setUrls(List<URI> urls) {
    this.urls = urls;
  }
}
