/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.component;

import de.fraunhofer.iese.ind2uce.api.component.description.MethodInterfaceDescription;
import de.fraunhofer.iese.ind2uce.api.component.identifier.ComponentId;

import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.hibernate.annotations.Fetch;
import org.hibernate.annotations.FetchMode;

import java.net.URI;
import java.util.Arrays;
import java.util.List;

import javax.persistence.CascadeType;
import javax.persistence.Entity;
import javax.persistence.JoinColumn;
import javax.persistence.JoinColumns;
import javax.persistence.OneToMany;

/**
 * A description of a concrete IND2UCE framework component. A component is of a
 * certain {@link ComponentType} and contains information about the its identity
 * ({@link ComponentId}). In addition, it may contain information about the
 * location ( {@link URI}) and the usage ( {@link MethodInterfaceDescription}).
 *
 * @author Fraunhofer IESE
 */
@Entity
public class Component extends ComponentBase {

  /**
   * The constant serialVersionUID.
   */
  private static final long serialVersionUID = -4320938758386608680L;

  /**
   * A description of the methods provided by the {@link Component} (i.e., its
   * Interface).
   */
  @OneToMany(cascade = CascadeType.ALL, orphanRemoval = true)
  @JoinColumns({
      @JoinColumn(referencedColumnName = "component_type", name = "component_type"), @JoinColumn(referencedColumnName = "identifier", name = "identifier"),
      @JoinColumn(referencedColumnName = "scope", name = "scope")
  })
  @Fetch(FetchMode.SUBSELECT)
  private List<MethodInterfaceDescription> componentInterfaces;

  /**
   * Default constructor for JPA.
   */
  Component() {
    super();
  }

  /**
   * Constructs a {@link Component} without URLs or ComponentInterface.
   *
   * @param id the unique ID of the {@link Component}.
   */
  public Component(ComponentId id) {
    this(id, null);
  }

  /**
   * Constructs a {@link Component}, including its location ( {@link URI}) and
   * its usage ({@link MethodInterfaceDescription}).
   *
   * @param id the unique ID of the {@link Component}
   * @param urls a list of {@link URI}s that can be used to communicate with the
   *          {@link Component}
   * @param componentInterfaces the component interfaces
   */
  public Component(ComponentId id, List<URI> urls, List<MethodInterfaceDescription> componentInterfaces) {
    super(id, urls);
    this.componentInterfaces = componentInterfaces;
  }

  /**
   * Constructs a {@link Component}, including its location ( {@link URI}) and
   * its usage ({@link MethodInterfaceDescription}).
   *
   * @param id the unique ID of the {@link Component}
   * @param urls a list of {@link URI}s that can be used to communicate with the
   *          {@link Component}
   * @param componentInterface a list of methods provided by the
   *          {@link Component} (i.e., its Interface)
   */
  public Component(ComponentId id, List<URI> urls, MethodInterfaceDescription... componentInterface) {
    this(id, urls, Arrays.asList(componentInterface));
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  public boolean equals(Object obj) {
    if (!(obj instanceof Component)) {
      return false;
    }
    final Component other = (Component)obj;
    if (this.id != null) {
      return this.id.equals(other.getId());
    }

    return super.equals(obj);
  }

  /**
   * Gets the a description of the methods provided by the {@link Component}.
   *
   * @return a description of the methods provided by the {@link Component}
   *         (i.e., its Interface)
   */
  public List<MethodInterfaceDescription> getComponentInterfaces() {
    return this.componentInterfaces;
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#hashCode()
   */
  @Override
  public int hashCode() {
    final HashCodeBuilder builder = new HashCodeBuilder(17, 31);
    builder.append(this.id);
    return builder.toHashCode();
  }

  /**
   * Sets the a description of the methods provided by the {@link Component}.
   *
   * @param componentInterface the list of methods provided by the
   *          {@link Component} (i.e., its Interface)
   */
  public void setComponentInterfaces(List<MethodInterfaceDescription> componentInterface) {
    this.componentInterfaces = componentInterface;
  }
}
