/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.common;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonPrimitive;
import com.google.gson.TypeAdapter;
import com.google.gson.TypeAdapterFactory;
import com.google.gson.internal.Streams;
import com.google.gson.reflect.TypeToken;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonWriter;

import java.io.IOException;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * Adapts values whose runtime type may differ from their declaration type. This
 * is necessary when a field's type is not the same type that GSON should create
 * when deserializing that field. For example, consider these types:
 *
 * <pre>
 * {
 *   &#64;code
 *   abstract class Shape {
 *     int x;
 *
 *     int y;
 *   }
 *   class Circle extends Shape {
 *     int radius;
 *   }
 *   class Rectangle extends Shape {
 *     int width;
 *
 *     int height;
 *   }
 *   class Diamond extends Shape {
 *     int width;
 *
 *     int height;
 *   }
 *   class Drawing {
 *     Shape bottomShape;
 *
 *     Shape topShape;
 *   }
 * }
 * </pre>
 * <p>
 * Without additional type information, the serialized JSON is ambiguous. Is the
 * bottom shape in this drawing a rectangle or a diamond?
 *
 * <pre>
 *    {@code
 *   {
 *     "bottomShape": {
 *       "width": 10,
 *       "height": 5,
 *       "x": 0,
 *       "y": 0
 *     },
 *     "topShape": {
 *       "radius": 2,
 *       "x": 4,
 *       "y": 1
 *     }
 *   }}
 * </pre>
 *
 * This class addresses this problem by adding type information to the
 * serialized JSON and honoring that type information when the JSON is
 * deserialized:
 *
 * <pre>
 *    {@code
 *   {
 *     "bottomShape": {
 *       "type": "Diamond",
 *       "width": 10,
 *       "height": 5,
 *       "x": 0,
 *       "y": 0
 *     },
 *     "topShape": {
 *       "type": "Circle",
 *       "radius": 2,
 *       "x": 4,
 *       "y": 1
 *     }
 *   }}
 * </pre>
 *
 * Both the type field name ({@code "type"}) and the type labels ({@code
 * "Rectangle"}) are configurable.
 * <h3>Registering Types</h3> Create a {@code RuntimeTypeAdapterFactory} by
 * passing the base type and type field name to the {@link #of} factory method.
 * If you don't supply an explicit type field name, {@code "type"} will be used.
 *
 * <pre>
 * {
 *   &#64;code
 *   RuntimeTypeAdapterFactory&lt;Shape$gt; shapeAdapterFactory = RuntimeTypeAdapterFactory.of(Shape.class, "type");
 * }
 * </pre>
 *
 * Next register all of your subtypes. Every subtype must be explicitly
 * registered. This protects your application from injection attacks. If you
 * don't supply an explicit type label, the type's simple name will be used.
 *
 * <pre>
 *    {@code
 *   shapeAdapter.registerSubtype(Rectangle.class, "Rectangle");
 *   shapeAdapter.registerSubtype(Circle.class, "Circle");
 *   shapeAdapter.registerSubtype(Diamond.class, "Diamond");
 * }
 * </pre>
 *
 * Finally, register the type adapter factory in your application's GSON
 * builder:
 *
 * <pre>
 * {
 *   &#64;code
 *   Gson gson = new GsonBuilder().registerTypeAdapterFactory(shapeAdapterFactory).create();
 * }
 * </pre>
 *
 * Like {@code GsonBuilder}, this API supports chaining:
 *
 * <pre>
 * {
 *   &#64;code
 *   RuntimeTypeAdapterFactory shapeAdapterFactory = RuntimeTypeAdapterFactory.of(Shape.class).registerSubtype(Rectangle.class).registerSubtype(Circle.class).registerSubtype(Diamond.class);
 * }
 * </pre>
 *
 * @param <T> the generic type
 */
public final class RuntimeTypeAdapterFactory<T> implements TypeAdapterFactory {

  /**
   * The base type.
   */
  private final Class<?> baseType;

  /**
   * The type field name.
   */
  private final String typeFieldName;

  /**
   * The label to subtype.
   */
  private final Map<String, Class<?>> labelToSubtype = new LinkedHashMap<>();

  /**
   * The subtype to label.
   */
  private final Map<Class<?>, String> subtypeToLabel = new LinkedHashMap<>();

  /**
   * Instantiates a new runtime type adapter factory.
   *
   * @param baseType the base type
   * @param typeFieldName the type field name
   */
  private RuntimeTypeAdapterFactory(Class<?> baseType, String typeFieldName) {
    if (typeFieldName == null || baseType == null) {
      throw new NullPointerException();
    }
    this.baseType = baseType;
    this.typeFieldName = typeFieldName;
  }

  /**
   * Creates a new runtime type adapter for {@code baseType} using
   * {@code "type"} as the type field name.
   *
   * @param <T> the generic type
   * @param baseType the base type
   * @return the runtime type adapter factory
   */
  public static <T> RuntimeTypeAdapterFactory<T> of(Class<T> baseType) {
    return new RuntimeTypeAdapterFactory<>(baseType, "type");
  }

  /**
   * Creates a new runtime type adapter using for {@code baseType} using {@code
   * typeFieldName} as the type field name. Type field names are case sensitive.
   *
   * @param <T> the generic type
   * @param baseType the base type
   * @param typeFieldName the type field name
   * @return the runtime type adapter factory
   */
  public static <T> RuntimeTypeAdapterFactory<T> of(Class<T> baseType, String typeFieldName) {
    return new RuntimeTypeAdapterFactory<>(baseType, typeFieldName);
  }

  /*
   * (non-Javadoc)
   * @see com.google.gson.TypeAdapterFactory#create(com.google.gson.Gson,
   * com.google.gson.reflect.TypeToken)
   */
  @Override
  public <R> TypeAdapter<R> create(Gson gson, TypeToken<R> type) {
    if (type.getRawType() != this.baseType) {
      return null;
    }

    final Map<String, TypeAdapter<?>> labelToDelegate = new LinkedHashMap<>();
    final Map<Class<?>, TypeAdapter<?>> subtypeToDelegate = new LinkedHashMap<>();
    for (final Map.Entry<String, Class<?>> entry : this.labelToSubtype.entrySet()) {
      final TypeAdapter<?> delegate = gson.getDelegateAdapter(this, TypeToken.get(entry.getValue()));
      labelToDelegate.put(entry.getKey(), delegate);
      subtypeToDelegate.put(entry.getValue(), delegate);
    }

    return new TypeAdapter<R>() {
      @Override
      public R read(JsonReader in) throws IOException {
        final JsonElement jsonElement = Streams.parse(in);
        final JsonElement labelJsonElement = jsonElement.getAsJsonObject().remove(RuntimeTypeAdapterFactory.this.typeFieldName);
        if (labelJsonElement == null) {
          throw new JsonParseException("cannot deserialize " + RuntimeTypeAdapterFactory.this.baseType + " because it does not define a field named " + RuntimeTypeAdapterFactory.this.typeFieldName);
        }
        final String label = labelJsonElement.getAsString();
        @SuppressWarnings("unchecked") // registration requires that subtype
                                       // extends T
        final TypeAdapter<R> delegate = (TypeAdapter<R>)labelToDelegate.get(label);
        if (delegate == null) {
          throw new JsonParseException("cannot deserialize " + RuntimeTypeAdapterFactory.this.baseType + " subtype named " + label + "; did you forget to register a subtype?");
        }
        return delegate.fromJsonTree(jsonElement);
      }

      @Override
      public void write(JsonWriter out, R value) throws IOException {
        final Class<?> srcType = value.getClass();
        final String label = RuntimeTypeAdapterFactory.this.subtypeToLabel.get(srcType);
        @SuppressWarnings("unchecked") // registration requires that subtype
                                       // extends T
        final TypeAdapter<R> delegate = (TypeAdapter<R>)subtypeToDelegate.get(srcType);
        if (delegate == null) {
          throw new JsonParseException("cannot serialize " + srcType.getName() + "; did you forget to register a subtype?");
        }
        final JsonObject jsonObject = delegate.toJsonTree(value).getAsJsonObject();
        if (jsonObject.has(RuntimeTypeAdapterFactory.this.typeFieldName)) {
          throw new JsonParseException("cannot serialize " + srcType.getName() + " because it already defines a field named " + RuntimeTypeAdapterFactory.this.typeFieldName);
        }
        final JsonObject clone = new JsonObject();
        clone.add(RuntimeTypeAdapterFactory.this.typeFieldName, new JsonPrimitive(label));
        for (final Map.Entry<String, JsonElement> e : jsonObject.entrySet()) {
          clone.add(e.getKey(), e.getValue());
        }
        Streams.write(clone, out);
      }
    }.nullSafe();
  }

  /**
   * Registers {@code type} identified by its {@link Class#getSimpleName simple
   * name}. Labels are case sensitive.
   *
   * @param type the type
   * @return the runtime type adapter factory
   * @throws IllegalArgumentException if either {@code type} or its simple name
   *           have already been registered on this type adapter.
   */
  public RuntimeTypeAdapterFactory<T> registerSubtype(Class<? extends T> type) {
    return this.registerSubtype(type, type.getSimpleName());
  }

  /**
   * Registers {@code type} identified by {@code label}. Labels are case
   * sensitive.
   *
   * @param type the type
   * @param label the label
   * @return the runtime type adapter factory
   * @throws IllegalArgumentException if either {@code type} or {@code label}
   *           have already been registered on this type adapter.
   */
  public RuntimeTypeAdapterFactory<T> registerSubtype(Class<? extends T> type, String label) {
    if (type == null || label == null) {
      throw new NullPointerException();
    }
    if (this.subtypeToLabel.containsKey(type) || this.labelToSubtype.containsKey(label)) {
      throw new IllegalArgumentException("types and labels must be unique");
    }
    this.labelToSubtype.put(label, type);
    this.subtypeToLabel.put(type, label);
    return this;
  }
}
