/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.api.common;

import de.fraunhofer.iese.ind2uce.api.component.description.ClassTypeDescription;
import de.fraunhofer.iese.ind2uce.api.component.description.MethodInterfaceDescription;
import de.fraunhofer.iese.ind2uce.api.component.description.TypeDescription;
import de.fraunhofer.iese.ind2uce.api.policy.parameter.DataObject;
import de.fraunhofer.iese.ind2uce.logger.LoggerFactory;

import com.google.gson.ExclusionStrategy;
import com.google.gson.FieldAttributes;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;

import java.io.Serializable;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.List;

/**
 * Common super class for all other IND2UCE classes.
 *
 * @author Fraunhofer IESE
 */
public abstract class Ind2uceEntity implements Serializable {

  /** The unknown classes. */
  private static List<String> unknownClasses = new ArrayList<>();

  /** The logger. */
  private static final org.slf4j.Logger LOG = LoggerFactory.getLogger(Ind2uceEntity.class);

  /**
   * The Constant serialVersionUID.
   */
  private static final long serialVersionUID = 4880779801528117995L;

  /**
   * The Constant HIDE_EXCLUSION.
   */
  private static final ExclusionStrategy HIDE_EXCLUSION = new ExclusionStrategy() {
    @Override
    public boolean shouldSkipClass(Class<?> aClass) {
      return false;
    }

    @Override
    public boolean shouldSkipField(FieldAttributes fieldAttributes) {
      final Hide hide = fieldAttributes.getAnnotation(Hide.class);
      return hide != null;
    }
  };

  /**
   * The Constant DATAOBJECT_DESERIALIZER.
   */
  private static final JsonDeserializer<DataObject<?>> DATAOBJECT_DESERIALIZER = new JsonDeserializer<DataObject<?>>() {
    /**
     * private method creates an object from an unknown class
     *
     * @param value the value
     * @param type the type
     * @param isComplex represents whether the object is complex
     * @return
     */
    private DataObject<?> createObjectFromUnkownClass(final JsonElement value, final String type, final boolean isComplex) {
      String result;

      if (!isComplex || isComplex && Object.class.getCanonicalName().equals(type)) {
        result = value.toString();
      } else {
        result = value.getAsString();
      }

      @SuppressWarnings({
          "unchecked", "rawtypes"
      })
      final DataObject o = new DataObject(result);
      o.setType(type);
      o.setComplex(true);
      return o;
    }

    @SuppressWarnings({
        "unchecked", "rawtypes"
    })
    @Override
    public DataObject<?> deserialize(JsonElement arg0, Type arg1, JsonDeserializationContext arg2) {

      final JsonElement value = arg0.getAsJsonObject().get("value");
      final String type = arg0.getAsJsonObject().get("type").getAsString();
      final boolean isComplex = arg0.getAsJsonObject().get("isComplex").getAsBoolean();

      if (unknownClasses.contains(type)) {
        return this.createObjectFromUnkownClass(value, type, isComplex);
      }
      try {
        final Object o = GSON_PLAIN.fromJson(value, Class.forName(type));

        return new DataObject(o, Class.forName(type));

      } catch (final ClassNotFoundException e) {
        LOG.warn("Deserialization failed. Adding " + type + " to the list of unknown classes", e);
        unknownClasses.add(type);

        return this.createObjectFromUnkownClass(value, type, isComplex);
      }
    }
  };

  /**
   * The Constant DATAOBJECT_DESERIALIZER.
   */
  private static final JsonSerializer<DataObject<?>> DATAOBJECT_SERIALIZER = new JsonSerializer<DataObject<?>>() {
    @Override
    public JsonElement serialize(DataObject<?> src, Type typeOfSrc, JsonSerializationContext context) {
      final JsonObject jsonObject = new JsonObject();
      jsonObject.add("value", GSON_PLAIN.toJsonTree(src.getValue()));
      jsonObject.addProperty("type", src.getTypeName());
      jsonObject.addProperty("isComplex", src.isComplex());
      return jsonObject;
    }
  };

  /** The Constant TYPE_DESCRIPTION_JSON_SERIALIZER. */
  private static final JsonSerializer<TypeDescription> TYPE_DESCRIPTION_JSON_SERIALIZER = new JsonSerializer<TypeDescription>() {
    @Override
    public JsonElement serialize(TypeDescription src, Type typeOfSrc, JsonSerializationContext context) {
      final JsonObject jsonObject = (JsonObject)GSON_PLAIN.toJsonTree(src);
      jsonObject.addProperty("type", src.getClass().getSimpleName());
      return jsonObject;
    }
  };

  /** The Constant TYPE_DESCRIPTION_JSON_DESERIALIZER. */
  private static final JsonDeserializer<TypeDescription> TYPE_DESCRIPTION_JSON_DESERIALIZER = new JsonDeserializer<TypeDescription>() {
    @Override
    public TypeDescription deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context) throws JsonParseException {
      try {
        final String type = json.getAsJsonObject().get("type").getAsString();
        switch (type) {
          case "TypeDescription":
            return GSON_PLAIN.fromJson(json, TypeDescription.class);
          case "ClassTypeDescription":
            return GSON_PLAIN.fromJson(json, ClassTypeDescription.class);
          default:
            return null;
        }
      } catch (final Exception e) {
        LOG.debug("Using default GSON", e);
        return GSON_PLAIN.fromJson(json, TypeDescription.class);
      }
    }
  };

  /** The Constant INTERFACE_DESCRIPTION_ADAPTER. */
  private static final RuntimeTypeAdapterFactory<MethodInterfaceDescription> INTERFACE_DESCRIPTION_ADAPTER = RuntimeTypeAdapterFactory.of(MethodInterfaceDescription.class)
      .registerSubtype(MethodInterfaceDescription.class);

  /**
   * The Constant GSON_PLAIN.
   */
  protected static final Gson GSON_PLAIN = new GsonBuilder().create();

  /**
   * The Constant GSON_DEFAULT.
   */
  protected static final Gson GSON_DEFAULT = new GsonBuilder().addSerializationExclusionStrategy(HIDE_EXCLUSION).registerTypeAdapter(DataObject.class, DATAOBJECT_DESERIALIZER)
      .registerTypeAdapter(DataObject.class, DATAOBJECT_SERIALIZER).registerTypeAdapter(TypeDescription.class, TYPE_DESCRIPTION_JSON_SERIALIZER)
      .registerTypeAdapter(TypeDescription.class, TYPE_DESCRIPTION_JSON_DESERIALIZER).registerTypeAdapter(ClassTypeDescription.class, TYPE_DESCRIPTION_JSON_SERIALIZER)
      .registerTypeAdapter(ClassTypeDescription.class, TYPE_DESCRIPTION_JSON_DESERIALIZER).registerTypeAdapterFactory(INTERFACE_DESCRIPTION_ADAPTER).create();

  /**
   * The Constant GSON_PRETTY.
   */
  protected static final Gson GSON_PRETTY = new GsonBuilder().setPrettyPrinting().addSerializationExclusionStrategy(HIDE_EXCLUSION).registerTypeAdapter(DataObject.class, DATAOBJECT_DESERIALIZER)
      .registerTypeAdapter(DataObject.class, DATAOBJECT_SERIALIZER).registerTypeAdapterFactory(INTERFACE_DESCRIPTION_ADAPTER).create();

  /**
   * The version.
   */
  @Hide
  protected String version = this.getClass().getPackage().getImplementationVersion();

  /**
   * Deserializes an Ind2uceEntity from JSON.
   *
   * @param <T> the generic type (sub class of {@link Ind2uceEntity})
   * @param json the serialized object in JSON notation
   * @param clazz the generic type (sub class of {@link Ind2uceEntity})
   * @return the deserialized {@link Ind2uceEntity}
   */
  public static <T extends Ind2uceEntity> T fromJson(String json, Class<T> clazz) {
    return GSON_PRETTY.fromJson(json, clazz);
  }

  /**
   * Gets the gson.
   *
   * @return the gson
   */
  public static Gson getGson() {
    return new GsonBuilder().setPrettyPrinting().addSerializationExclusionStrategy(HIDE_EXCLUSION).registerTypeAdapter(DataObject.class, DATAOBJECT_DESERIALIZER)
        .registerTypeAdapter(TypeDescription.class, TYPE_DESCRIPTION_JSON_SERIALIZER).registerTypeAdapter(TypeDescription.class, TYPE_DESCRIPTION_JSON_DESERIALIZER)
        .registerTypeAdapter(ClassTypeDescription.class, TYPE_DESCRIPTION_JSON_SERIALIZER).registerTypeAdapter(ClassTypeDescription.class, TYPE_DESCRIPTION_JSON_DESERIALIZER)
        .registerTypeAdapterFactory(INTERFACE_DESCRIPTION_ADAPTER).create();
  }

  /**
   * Gets the version.
   *
   * @return the version
   */
  public String getVersion() {
    return this.version;
  }

  /**
   * Serializes an Ind2uceEntity to JSON.
   *
   * @param pretty if true, pretty printing is enabled
   * @return the JSON serialized {@link Ind2uceEntity}
   */
  public String toJson(boolean pretty) {
    if (pretty) {
      return GSON_PRETTY.toJson(this);
    } else {
      return GSON_DEFAULT.toJson(this);
    }
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#toString()
   */
  @Override
  public String toString() {
    return this.toJson(false);
  }

}
