/*
 * Decompiled with CFR 0.152.
 */
package crypto.cryslhandler;

import com.google.common.base.CharMatcher;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.inject.Injector;
import crypto.cryslhandler.CrySLReaderUtils;
import crypto.cryslhandler.ExceptionsReader;
import crypto.cryslhandler.StateMachineGraphBuilder;
import crypto.exceptions.CryptoAnalysisException;
import crypto.interfaces.ICrySLPredicateParameter;
import crypto.interfaces.ISLConstraint;
import crypto.rules.CrySLArithmeticConstraint;
import crypto.rules.CrySLComparisonConstraint;
import crypto.rules.CrySLCondPredicate;
import crypto.rules.CrySLConstraint;
import crypto.rules.CrySLForbiddenMethod;
import crypto.rules.CrySLMethod;
import crypto.rules.CrySLObject;
import crypto.rules.CrySLPredicate;
import crypto.rules.CrySLRule;
import crypto.rules.CrySLSplitter;
import crypto.rules.CrySLValueConstraint;
import crypto.rules.StateMachineGraph;
import crypto.rules.StateNode;
import crypto.rules.TransitionEdge;
import de.darmstadt.tu.crossing.CrySLStandaloneSetup;
import de.darmstadt.tu.crossing.crySL.AlternativeRequiredPredicates;
import de.darmstadt.tu.crossing.crySL.BuiltinPredicate;
import de.darmstadt.tu.crossing.crySL.ConditionalPredicate;
import de.darmstadt.tu.crossing.crySL.Constraint;
import de.darmstadt.tu.crossing.crySL.ConstraintsBlock;
import de.darmstadt.tu.crossing.crySL.Domainmodel;
import de.darmstadt.tu.crossing.crySL.EnsuresBlock;
import de.darmstadt.tu.crossing.crySL.Event;
import de.darmstadt.tu.crossing.crySL.EventsBlock;
import de.darmstadt.tu.crossing.crySL.ForbiddenBlock;
import de.darmstadt.tu.crossing.crySL.ForbiddenMethod;
import de.darmstadt.tu.crossing.crySL.LabeledMethodCall;
import de.darmstadt.tu.crossing.crySL.Literal;
import de.darmstadt.tu.crossing.crySL.LiteralExpression;
import de.darmstadt.tu.crossing.crySL.LiteralList;
import de.darmstadt.tu.crossing.crySL.NegatesBlock;
import de.darmstadt.tu.crossing.crySL.ObjectExpression;
import de.darmstadt.tu.crossing.crySL.ObjectOperation;
import de.darmstadt.tu.crossing.crySL.ObjectReference;
import de.darmstadt.tu.crossing.crySL.ObjectsBlock;
import de.darmstadt.tu.crossing.crySL.Operator;
import de.darmstadt.tu.crossing.crySL.Order;
import de.darmstadt.tu.crossing.crySL.OrderBlock;
import de.darmstadt.tu.crossing.crySL.Predicate;
import de.darmstadt.tu.crossing.crySL.PredicateCondition;
import de.darmstadt.tu.crossing.crySL.PredicateParameter;
import de.darmstadt.tu.crossing.crySL.RequiredPredicate;
import de.darmstadt.tu.crossing.crySL.RequiresBlock;
import de.darmstadt.tu.crossing.crySL.ThisPredicateParameter;
import de.darmstadt.tu.crossing.crySL.TimedPredicate;
import de.darmstadt.tu.crossing.crySL.WildcardPredicateParameter;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.lang3.NotImplementedException;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.common.types.access.impl.ClasspathTypeProvider;
import org.eclipse.xtext.diagnostics.Severity;
import org.eclipse.xtext.resource.XtextResource;
import org.eclipse.xtext.resource.XtextResourceSet;
import org.eclipse.xtext.util.CancelIndicator;
import org.eclipse.xtext.validation.CheckMode;
import org.eclipse.xtext.validation.IResourceValidator;
import org.eclipse.xtext.validation.Issue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CrySLModelReader {
    private static final Logger LOGGER = LoggerFactory.getLogger(CrySLModelReader.class);
    private StateMachineGraph smg = null;
    private JvmTypeReference currentClass;
    private final XtextResourceSet resourceSet;
    private final Injector injector;
    public static final String cryslFileEnding = ".crysl";
    private static final String THIS = "this";
    private static final String NULL = "null";
    private static final String UNDERSCORE = "_";
    private static final Set<String> buggedKeyRules = new HashSet<String>(Arrays.asList("java.security.Key", "javax.crypto.SecretKey", "java.security.PublicKey", "java.security.PrivateKey"));

    public CrySLModelReader() throws MalformedURLException {
        CrySLStandaloneSetup crySLStandaloneSetup = new CrySLStandaloneSetup();
        this.injector = crySLStandaloneSetup.createInjectorAndDoEMFRegistration();
        this.resourceSet = this.injector.getInstance(XtextResourceSet.class);
        String[] cp = System.getProperty("java.class.path").split(File.pathSeparator);
        URL[] classpath = new URL[cp.length];
        for (int i = 0; i < classpath.length; ++i) {
            classpath[i] = new File(cp[i]).toURI().toURL();
        }
        URLClassLoader ucl = new URLClassLoader(classpath);
        this.resourceSet.setClasspathURIContext(new URLClassLoader(classpath));
        new ClasspathTypeProvider(ucl, this.resourceSet, null, null);
        this.resourceSet.addLoadOption(XtextResource.OPTION_RESOLVE_ALL, Boolean.TRUE);
    }

    public CrySLRule readRule(InputStream stream, String virtualFileName) throws IllegalArgumentException, IOException, CryptoAnalysisException {
        if (!virtualFileName.endsWith(cryslFileEnding)) {
            throw new CryptoAnalysisException("The extension of " + virtualFileName + " does not match " + cryslFileEnding);
        }
        URI uri = URI.createURI(virtualFileName);
        Resource resource = this.resourceSet.getURIResourceMap().get(uri);
        if (resource == null) {
            resource = this.resourceSet.createResource(uri);
            resource.load(stream, Collections.EMPTY_MAP);
        }
        return this.createRuleFromResource(resource);
    }

    public CrySLRule readRule(File ruleFile) throws CryptoAnalysisException {
        String fileName = ruleFile.getName();
        if (!fileName.endsWith(cryslFileEnding)) {
            throw new CryptoAnalysisException("The extension of " + fileName + "  does not match " + cryslFileEnding);
        }
        Resource resource = this.resourceSet.getResource(URI.createFileURI(ruleFile.getAbsolutePath()), true);
        try {
            return this.createRuleFromResource(resource);
        }
        catch (Exception e) {
            System.err.println(e);
            return null;
        }
    }

    private boolean runValidator(Resource resource, Severity report) {
        IResourceValidator validator = this.injector.getInstance(IResourceValidator.class);
        List<Issue> issues = validator.validate(resource, CheckMode.ALL, CancelIndicator.NullImpl);
        boolean errorFound = false;
        for (Issue issue : issues) {
            switch (issue.getSeverity()) {
                case ERROR: {
                    if (report.compareTo(issue.getSeverity()) >= 0) {
                        LOGGER.error("{}:{}: {}", resource.getURI(), issue.getLineNumber(), issue.getMessage());
                    }
                    errorFound = true;
                    break;
                }
                case WARNING: {
                    if (report.compareTo(issue.getSeverity()) >= 0) {
                        LOGGER.warn("{}:{}: {}", resource.getURI(), issue.getLineNumber(), issue.getMessage());
                    }
                    errorFound = true;
                    break;
                }
                case INFO: {
                    if (report.compareTo(issue.getSeverity()) < 0) break;
                    LOGGER.info("{}:{}: {}", resource.getURI(), issue.getLineNumber(), issue.getMessage());
                    break;
                }
            }
        }
        return errorFound;
    }

    private CrySLRule createRuleFromResource(Resource resource) throws CryptoAnalysisException {
        if (resource == null) {
            throw new CryptoAnalysisException("Internal error creating a CrySL rule: 'resource parameter was null'.");
        }
        String currentClass = ((Domainmodel)resource.getContents().get(0)).getJavaType().getQualifiedName();
        if (this.runValidator(resource, Severity.WARNING)) {
            if (buggedKeyRules.contains(currentClass)) {
                LOGGER.info("Class " + currentClass + " is of type java.security.key. The call to 'getEncoded()' will be resolved manually.");
            } else {
                throw new CryptoAnalysisException("Skipping rule since it contains errors: " + resource.getURI());
            }
        }
        try {
            return this.createRuleFromDomainmodel((Domainmodel)resource.getContents().get(0));
        }
        catch (Exception e) {
            e.printStackTrace();
            throw new CryptoAnalysisException("An error occured while reading the rule " + resource.getURI(), e);
        }
    }

    private CrySLRule createRuleFromDomainmodel(Domainmodel model) throws CryptoAnalysisException {
        this.currentClass = model.getJavaType();
        String currentClass = this.currentClass.getQualifiedName();
        if (currentClass.equals("void")) {
            throw new CryptoAnalysisException("Class for the rule is not on the classpath.");
        }
        List<Map.Entry<String, String>> objects = this.getObjects(model.getObjects());
        List<CrySLForbiddenMethod> forbiddenMethods = this.getForbiddenMethods(model.getForbidden());
        EventsBlock eventsBlock = model.getEvents();
        OrderBlock orderBlock = model.getOrder();
        List<Event> events = this.changeDeclaringClass(this.currentClass, eventsBlock);
        Order order = orderBlock == null ? null : orderBlock.getOrder();
        this.smg = StateMachineGraphBuilder.buildSMG(order, events);
        ArrayList<ISLConstraint> constraints = Lists.newArrayList();
        constraints.addAll(this.getConstraints(model.getConstraints()));
        constraints.addAll(this.getRequiredPredicates(model.getRequires()));
        constraints.addAll(ExceptionsReader.getExceptionConstraints(eventsBlock));
        EnsuresBlock ensuresBlock = model.getEnsures();
        NegatesBlock negatesBlock = model.getNegates();
        ArrayList<CrySLPredicate> predicates = Lists.newArrayList();
        ArrayList<CrySLPredicate> negatedPredicates = Lists.newArrayList();
        predicates.addAll(this.getEnsuredPredicates(ensuresBlock));
        negatedPredicates.addAll(this.getNegatedPredicates(negatesBlock));
        return new CrySLRule(currentClass, objects, forbiddenMethods, this.smg, constraints, predicates, negatedPredicates);
    }

    private List<Event> changeDeclaringClass(JvmTypeReference currentClass, EventsBlock eventsBlock) {
        if (eventsBlock == null) {
            return Collections.emptyList();
        }
        return eventsBlock.getEvents().stream().map(event -> event instanceof LabeledMethodCall ? this.changeDeclaringClass(currentClass, (LabeledMethodCall)event) : event).collect(Collectors.toList());
    }

    private Event changeDeclaringClass(JvmTypeReference currentClass, LabeledMethodCall event) {
        event.getMethod().getMethod().setDeclaringType((JvmDeclaredType)currentClass.getType());
        return event;
    }

    private List<Map.Entry<String, String>> getObjects(ObjectsBlock objects) {
        if (objects == null) {
            return Collections.emptyList();
        }
        return objects.getDeclarations().parallelStream().map(CrySLReaderUtils::resolveObject).collect(Collectors.toList());
    }

    private List<CrySLForbiddenMethod> getForbiddenMethods(ForbiddenBlock forbidden) {
        if (forbidden == null) {
            return Collections.emptyList();
        }
        ArrayList<CrySLForbiddenMethod> forbiddenMethods = Lists.newArrayList();
        for (ForbiddenMethod method : forbidden.getForbiddenMethods()) {
            CrySLMethod cryslMethod = CrySLReaderUtils.toCrySLMethod(method);
            List<CrySLMethod> alternatives = CrySLReaderUtils.resolveEventToCryslMethods(method.getReplacement());
            forbiddenMethods.add(new CrySLForbiddenMethod(cryslMethod, false, alternatives));
        }
        return forbiddenMethods;
    }

    private List<CrySLPredicate> getEnsuredPredicates(EnsuresBlock ensures) {
        if (ensures == null) {
            return Collections.emptyList();
        }
        return this.getTimedPredicates(ensures.getEnsuredPredicates(), false);
    }

    private List<CrySLPredicate> getNegatedPredicates(NegatesBlock negates) {
        if (negates == null) {
            return Collections.emptyList();
        }
        return this.getTimedPredicates(negates.getNegatedPredicates(), true);
    }

    private List<CrySLPredicate> getTimedPredicates(List<? extends TimedPredicate> timedPredicates, boolean negate) {
        ArrayList<CrySLPredicate> predicates = new ArrayList<CrySLPredicate>(timedPredicates.size());
        for (TimedPredicate timedPredicate : timedPredicates) {
            Predicate predicate = timedPredicate.getPredicate();
            ISLConstraint constraint = timedPredicate instanceof ConditionalPredicate ? this.getPredicateCondition(timedPredicate) : null;
            List<ICrySLPredicateParameter> parameters = this.resolvePredicateParameters(predicate);
            String name = predicate.getName();
            if (timedPredicate.getAfter() == null) {
                predicates.add(new CrySLPredicate(null, name, parameters, (Boolean)negate, constraint));
                continue;
            }
            Set<StateNode> nodes = this.getStatesForMethods(CrySLReaderUtils.resolveEventToCryslMethods(timedPredicate.getAfter()));
            predicates.add(new CrySLCondPredicate(null, name, parameters, negate, nodes, constraint));
        }
        return predicates;
    }

    private List<ICrySLPredicateParameter> resolvePredicateParameters(Predicate predicate) {
        ArrayList<ICrySLPredicateParameter> parameters = new ArrayList<ICrySLPredicateParameter>(predicate.getParameters().size());
        for (PredicateParameter parameter : predicate.getParameters()) {
            if (parameter instanceof WildcardPredicateParameter) {
                parameters.add(new CrySLObject(UNDERSCORE, NULL));
                continue;
            }
            if (parameter instanceof ThisPredicateParameter) {
                parameters.add(new CrySLObject(THIS, this.currentClass.getQualifiedName()));
                continue;
            }
            parameters.add(this.getObjectExpressionValue(parameter.getValue()));
        }
        return parameters;
    }

    private CrySLObject getObjectExpressionValue(ObjectExpression expression) {
        if (expression instanceof ObjectReference) {
            return this.getObjectExpressionValue((ObjectReference)expression);
        }
        if (expression instanceof ObjectOperation) {
            return this.getObjectExpressionValue((ObjectOperation)expression);
        }
        return null;
    }

    private CrySLObject getObjectExpressionValue(ObjectReference reference) {
        return CrySLReaderUtils.toCrySLObject(reference.getObject());
    }

    private CrySLObject getObjectExpressionValue(ObjectOperation operation) {
        String type = operation.getObject().getType().getQualifiedName();
        String name = operation.getObject().getName();
        switch (operation.getFn()) {
            case ALG: {
                return new CrySLObject(name, type, new CrySLSplitter(0, "/"));
            }
            case MODE: {
                return new CrySLObject(name, type, new CrySLSplitter(1, "/"));
            }
            case PAD: {
                return new CrySLObject(name, type, new CrySLSplitter(2, "/"));
            }
            case PART: {
                int index = Integer.parseInt(operation.getIndex());
                String split = operation.getSplit();
                return new CrySLObject(name, type, new CrySLSplitter(index, split));
            }
            case ELEMENTS: {
                return CrySLReaderUtils.toCrySLObject(operation.getObject());
            }
        }
        return null;
    }

    private ISLConstraint getPredicateCondition(ConditionalPredicate predicate) {
        PredicateCondition condition = predicate.getCondition();
        if (condition instanceof Constraint) {
            return this.getConstraint((Constraint)condition);
        }
        if (condition instanceof Predicate) {
            return this.getPredicate((Predicate)condition);
        }
        return null;
    }

    private CrySLPredicate getPredicate(Predicate predicate) {
        return this.getPredicate(predicate, false, null);
    }

    private CrySLPredicate getPredicate(Predicate predicate, boolean negate, ISLConstraint constraint) {
        List<ICrySLPredicateParameter> variables = this.resolvePredicateParameters(predicate);
        return new CrySLPredicate(null, predicate.getName(), variables, (Boolean)negate, constraint);
    }

    private List<ISLConstraint> getRequiredPredicates(RequiresBlock requiresBlock) {
        if (requiresBlock == null) {
            return Collections.emptyList();
        }
        ArrayList<ISLConstraint> predicates = new ArrayList<ISLConstraint>();
        EList<AlternativeRequiredPredicates> requiredPredicates = requiresBlock.getRequiredPredicates();
        for (AlternativeRequiredPredicates alternativePredicates : requiredPredicates) {
            List alternatives = alternativePredicates.getAlternatives().parallelStream().map(this::getRequiredPredicate).collect(Collectors.toList());
            ISLConstraint predicate = (ISLConstraint)alternatives.get(0);
            for (int i = 1; i < alternatives.size(); ++i) {
                predicate = new CrySLConstraint((ISLConstraint)alternatives.get(i), predicate, CrySLConstraint.LogOps.or);
            }
            predicates.add(predicate);
        }
        return predicates;
    }

    private CrySLPredicate getRequiredPredicate(RequiredPredicate predicate) {
        ISLConstraint constraint = this.getPredicateCondition(predicate);
        boolean negate = predicate.isNegated();
        return this.getPredicate(predicate.getPredicate(), negate, constraint);
    }

    private List<ISLConstraint> getConstraints(ConstraintsBlock constraintsBlock) {
        if (constraintsBlock == null) {
            return Collections.emptyList();
        }
        return constraintsBlock.getConstraints().parallelStream().map(this::getConstraint).collect(Collectors.toList());
    }

    private ISLConstraint getConstraint(Constraint constraint) {
        if (constraint instanceof LiteralExpression) {
            return this.getLiteralExpression((LiteralExpression)constraint);
        }
        switch (constraint.getOp()) {
            case NOT: {
                throw new NotImplementedException("The NOT operator is not implemented.");
            }
            case IMPLY: 
            case OR: 
            case AND: {
                ISLConstraint left = this.getConstraint(constraint.getLeft());
                ISLConstraint right = this.getConstraint(constraint.getRight());
                CrySLConstraint.LogOps logOp = CrySLReaderUtils.logOpFromOperator(constraint.getOp()).get();
                return new CrySLConstraint(left, right, logOp);
            }
            case EQUAL: 
            case UNEQUAL: 
            case GREATER: 
            case GREATER_OR_EQUAL: 
            case LESS: 
            case LESS_OR_EQUAL: {
                CrySLComparisonConstraint.CompOp compOp = CrySLReaderUtils.compOpFromOperator(constraint.getOp()).get();
                CrySLArithmeticConstraint left = this.coerceConstraintToArithmeticConstraint(this.getConstraint(constraint.getLeft()));
                CrySLArithmeticConstraint right = this.coerceConstraintToArithmeticConstraint(this.getConstraint(constraint.getRight()));
                return new CrySLComparisonConstraint(left, right, compOp);
            }
            case TIMES: 
            case DIVIDE: {
                throw new NotImplementedException("The multiplication operators are not implemented.");
            }
            case PLUS: 
            case MINUS: 
            case MODULO: {
                ISLConstraint left = this.getConstraint(constraint.getLeft());
                ISLConstraint right = this.getConstraint(constraint.getRight());
                CrySLArithmeticConstraint.ArithOp arithOp = CrySLReaderUtils.arithOpFromOperator(constraint.getOp()).get();
                return new CrySLArithmeticConstraint(left, right, arithOp);
            }
            case IN: {
                CrySLObject left;
                CrySLObject crySLObject = constraint.getLeft() instanceof ObjectExpression ? this.getObjectExpressionValue((ObjectExpression)constraint.getLeft()) : (left = constraint.getLeft() instanceof Literal ? CrySLReaderUtils.toCrySLObject((Literal)constraint) : null);
                if (left == null) {
                    throw new IllegalArgumentException("lhs of an IN expression must be an Object or an Operation thereon.");
                }
                LiteralList right = (LiteralList)constraint.getRight();
                List<String> values = right.getElements().stream().map(Literal::getValue).collect(Collectors.toList());
                return new CrySLValueConstraint(left, values);
            }
        }
        return null;
    }

    private CrySLArithmeticConstraint coerceConstraintToArithmeticConstraint(ISLConstraint constraint) {
        if (constraint instanceof CrySLArithmeticConstraint) {
            return (CrySLArithmeticConstraint)constraint;
        }
        if (constraint instanceof CrySLPredicate) {
            return this.makeArithmeticConstraint((CrySLPredicate)constraint);
        }
        throw new ClassCastException("Cant coerce `" + constraint.toString() + "` into ArithmeticExpression");
    }

    private ISLConstraint getLiteralExpression(LiteralExpression expression) {
        if (expression instanceof BuiltinPredicate) {
            return this.getBuiltinPredicate((BuiltinPredicate)expression);
        }
        if (expression instanceof Literal) {
            return this.makeConstraintFromObject(CrySLReaderUtils.toCrySLObject((Literal)expression));
        }
        if (expression instanceof ObjectExpression) {
            return this.makeConstraintFromObject(this.getObjectExpressionValue((ObjectExpression)expression));
        }
        return null;
    }

    private ISLConstraint makeConstraintFromObject(ICrySLPredicateParameter object) {
        return this.makeArithmeticConstraint(object);
    }

    private CrySLArithmeticConstraint makeArithmeticConstraint(ICrySLPredicateParameter object) {
        CrySLObject zero = new CrySLObject("0", "int");
        CrySLArithmeticConstraint.ArithOp plus = CrySLReaderUtils.arithOpFromOperator(Operator.PLUS).get();
        return new CrySLArithmeticConstraint(object, zero, plus);
    }

    private Set<StateNode> getStatesForMethods(List<CrySLMethod> condition) {
        HashSet<StateNode> predicateGenerationNodes = new HashSet<StateNode>();
        if (condition.size() == 0) {
            return predicateGenerationNodes;
        }
        for (TransitionEdge transition : this.smg.getAllTransitions()) {
            if (!transition.getLabel().containsAll(condition)) continue;
            Set<StateNode> reachableNodes = this.getAllReachableNodes(transition.getRight(), Sets.newHashSet());
            predicateGenerationNodes.addAll(reachableNodes);
        }
        return predicateGenerationNodes;
    }

    private Set<StateNode> getAllReachableNodes(StateNode startNode, Set<StateNode> visited) {
        if (visited.contains(startNode)) {
            return visited;
        }
        visited.add(startNode);
        for (TransitionEdge edge : this.smg.getAllOutgoingEdges(startNode)) {
            Set<StateNode> reachableNodes = this.getAllReachableNodes(edge.getRight(), visited);
            visited.addAll(reachableNodes);
        }
        return visited;
    }

    private ISLConstraint getBuiltinPredicate(BuiltinPredicate builtinPredicate) {
        List<ICrySLPredicateParameter> parameters;
        String name = builtinPredicate.getPredicate().getLiteral();
        boolean negated = false;
        switch (builtinPredicate.getPredicate()) {
            case NO_CALL_TO: 
            case CALL_TO: {
                parameters = CrySLReaderUtils.resolveEventToPredicateParameters(builtinPredicate.getEvent());
                break;
            }
            case INSTANCE_OF: 
            case NEVER_TYPE_OF: {
                parameters = Lists.newArrayList(CrySLReaderUtils.toCrySLObject(builtinPredicate.getObject()), new CrySLObject(builtinPredicate.getType().getQualifiedName(), NULL));
                break;
            }
            case NOT_HARD_CODED: 
            case LENGTH: {
                CrySLObject object = CrySLReaderUtils.toCrySLObject(builtinPredicate.getObject());
                parameters = Collections.singletonList(object);
                break;
            }
            default: {
                parameters = Collections.emptyList();
            }
        }
        return new CrySLPredicate(null, name, parameters, negated);
    }

    public static Set<String> getBuggedKeyRules() {
        return buggedKeyRules;
    }

    public static String filterQuotes(String dirty) {
        return CharMatcher.anyOf("\"").removeFrom(dirty);
    }
}

