/*
 * Copyright (c) 2019, Fraunhofer AISEC. All rights reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *                    $$$$$$\  $$$$$$$\   $$$$$$\
 *                   $$  __$$\ $$  __$$\ $$  __$$\
 *                   $$ /  \__|$$ |  $$ |$$ /  \__|
 *                   $$ |      $$$$$$$  |$$ |$$$$\
 *                   $$ |      $$  ____/ $$ |\_$$ |
 *                   $$ |  $$\ $$ |      $$ |  $$ |
 *                   \$$$$$   |$$ |      \$$$$$   |
 *                    \______/ \__|       \______/
 *
 */

package de.fraunhofer.aisec.cpg.graph;

import de.fraunhofer.aisec.cpg.graph.HasType.TypeListener;
import de.fraunhofer.aisec.cpg.graph.Type.Origin;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import org.apache.commons.lang3.builder.ToStringBuilder;

/**
 * Represents an expression containing a ternary operator: var x = condition ? valueIfTrue :
 * valueIfFalse;
 */
public class ConditionalExpression extends Expression implements TypeListener {

  @SubGraph("AST")
  private Expression condition;

  @SubGraph("AST")
  private Expression thenExpr;

  @SubGraph("AST")
  private Expression elseExpr;

  public Expression getCondition() {
    return condition;
  }

  public void setCondition(Expression condition) {
    this.condition = condition;
  }

  public Expression getThenExpr() {
    return thenExpr;
  }

  public void setThenExpr(Expression thenExpr) {
    if (this.thenExpr != null) {
      this.thenExpr.unregisterTypeListener(this);
      this.removePrevDFG(this.thenExpr);
    }
    this.thenExpr = thenExpr;
    if (thenExpr != null) {
      thenExpr.registerTypeListener(this);
      this.addPrevDFG(thenExpr);
    }
  }

  public Expression getElseExpr() {
    return elseExpr;
  }

  public void setElseExpr(Expression elseExpr) {
    if (this.elseExpr != null) {
      this.elseExpr.unregisterTypeListener(this);
      this.removePrevDFG(this.elseExpr);
    }
    this.elseExpr = elseExpr;
    if (elseExpr != null) {
      elseExpr.registerTypeListener(this);
      this.addPrevDFG(elseExpr);
    }
  }

  @Override
  public void typeChanged(HasType src, HasType root, Type oldType) {
    Type previous = this.type;

    List<Type> types = new ArrayList<>();
    if (thenExpr != null && thenExpr.getType() != null) {
      types.add(thenExpr.getType());
    }
    if (elseExpr != null && elseExpr.getType() != null) {
      types.add(elseExpr.getType());
    }
    Set<Type> subTypes = new HashSet<>(getPossibleSubTypes());
    subTypes.remove(oldType);
    subTypes.addAll(types);

    Type alternative = !types.isEmpty() ? types.get(0) : null;
    setType(TypeManager.getInstance().getCommonType(types).orElse(alternative), root);
    setPossibleSubTypes(subTypes, root);

    if (!previous.equals(this.type)) {
      this.type.setTypeOrigin(Origin.DATAFLOW);
    }
  }

  @Override
  public void possibleSubTypesChanged(HasType src, HasType root, Set<Type> oldSubTypes) {
    Set<Type> subTypes = new HashSet<>(getPossibleSubTypes());
    subTypes.addAll(src.getPossibleSubTypes());
    setPossibleSubTypes(subTypes);
  }

  @Override
  public String toString() {
    return new ToStringBuilder(this, Node.TO_STRING_STYLE)
        .appendSuper(super.toString())
        .append("condition", condition)
        .append("thenExpr", thenExpr)
        .append("elseExpr", elseExpr)
        .build();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (!(o instanceof ConditionalExpression)) {
      return false;
    }
    ConditionalExpression that = (ConditionalExpression) o;
    return super.equals(that)
        && Objects.equals(condition, that.condition)
        && Objects.equals(thenExpr, that.thenExpr)
        && Objects.equals(elseExpr, that.elseExpr);
  }

  @Override
  public int hashCode() {
    return super.hashCode();
  }
}
