package de.flapdoodle.embed.process.config.store;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link FileSet.Entry}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableEntry.builder()}.
 * Use the static factory method to create immutable instances:
 * {@code ImmutableEntry.of()}.
 */
@SuppressWarnings({"all"})
public final class ImmutableEntry extends FileSet.Entry {
  private final FileType type;
  private final String destination;
  private final UncompiledPattern uncompiledMatchingPattern;

  private ImmutableEntry(
      FileType type,
      String destination,
      UncompiledPattern uncompiledMatchingPattern) {
    this.type = Objects.requireNonNull(type, "type");
    this.destination = Objects.requireNonNull(destination, "destination");
    this.uncompiledMatchingPattern = Objects.requireNonNull(uncompiledMatchingPattern, "uncompiledMatchingPattern");
  }

  private ImmutableEntry(
      ImmutableEntry original,
      FileType type,
      String destination,
      UncompiledPattern uncompiledMatchingPattern) {
    this.type = type;
    this.destination = destination;
    this.uncompiledMatchingPattern = uncompiledMatchingPattern;
  }

  /**
   * @return The value of the {@code type} attribute
   */
  @Override
  public FileType type() {
    return type;
  }

  /**
   * @return The value of the {@code destination} attribute
   */
  @Override
  public String destination() {
    return destination;
  }

  /**
   * @return The value of the {@code uncompiledMatchingPattern} attribute
   */
  @Override
  protected UncompiledPattern uncompiledMatchingPattern() {
    return uncompiledMatchingPattern;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link FileSet.Entry#type() type} attribute.
   * A value equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for type
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableEntry withType(FileType value) {
    FileType newValue = Objects.requireNonNull(value, "type");
    if (this.type == newValue) return this;
    return new ImmutableEntry(this, newValue, this.destination, this.uncompiledMatchingPattern);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link FileSet.Entry#destination() destination} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for destination
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableEntry withDestination(String value) {
    String newValue = Objects.requireNonNull(value, "destination");
    if (this.destination.equals(newValue)) return this;
    return new ImmutableEntry(this, this.type, newValue, this.uncompiledMatchingPattern);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link FileSet.Entry#uncompiledMatchingPattern() uncompiledMatchingPattern} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for uncompiledMatchingPattern
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableEntry withUncompiledMatchingPattern(UncompiledPattern value) {
    if (this.uncompiledMatchingPattern == value) return this;
    UncompiledPattern newValue = Objects.requireNonNull(value, "uncompiledMatchingPattern");
    return new ImmutableEntry(this, this.type, this.destination, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableEntry} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableEntry
        && equalTo(0, (ImmutableEntry) another);
  }

  private boolean equalTo(int synthetic, ImmutableEntry another) {
    return type.equals(another.type)
        && destination.equals(another.destination)
        && uncompiledMatchingPattern.equals(another.uncompiledMatchingPattern);
  }

  /**
   * Computes a hash code from attributes: {@code type}, {@code destination}, {@code uncompiledMatchingPattern}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + type.hashCode();
    h += (h << 5) + destination.hashCode();
    h += (h << 5) + uncompiledMatchingPattern.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code Entry} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Entry{"
        + "type=" + type
        + ", destination=" + destination
        + ", uncompiledMatchingPattern=" + uncompiledMatchingPattern
        + "}";
  }

  /**
   * Construct a new immutable {@code Entry} instance.
   * @param type The value for the {@code type} attribute
   * @param destination The value for the {@code destination} attribute
   * @param uncompiledMatchingPattern The value for the {@code uncompiledMatchingPattern} attribute
   * @return An immutable Entry instance
   */
  public static ImmutableEntry of(FileType type, String destination, UncompiledPattern uncompiledMatchingPattern) {
    return new ImmutableEntry(type, destination, uncompiledMatchingPattern);
  }

  /**
   * Creates an immutable copy of a {@link FileSet.Entry} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Entry instance
   */
  public static ImmutableEntry copyOf(FileSet.Entry instance) {
    if (instance instanceof ImmutableEntry) {
      return (ImmutableEntry) instance;
    }
    return ImmutableEntry.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableEntry ImmutableEntry}.
   * <pre>
   * ImmutableEntry.builder()
   *    .type(de.flapdoodle.embed.process.config.store.FileType) // required {@link FileSet.Entry#type() type}
   *    .destination(String) // required {@link FileSet.Entry#destination() destination}
   *    .uncompiledMatchingPattern(de.flapdoodle.embed.process.config.store.UncompiledPattern) // required {@link FileSet.Entry#uncompiledMatchingPattern() uncompiledMatchingPattern}
   *    .build();
   * </pre>
   * @return A new ImmutableEntry builder
   */
  public static ImmutableEntry.Builder builder() {
    return new ImmutableEntry.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableEntry ImmutableEntry}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private static final long INIT_BIT_TYPE = 0x1L;
    private static final long INIT_BIT_DESTINATION = 0x2L;
    private static final long INIT_BIT_UNCOMPILED_MATCHING_PATTERN = 0x4L;
    private long initBits = 0x7L;

    private FileType type;
    private String destination;
    private UncompiledPattern uncompiledMatchingPattern;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Entry} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(FileSet.Entry instance) {
      Objects.requireNonNull(instance, "instance");
      type(instance.type());
      destination(instance.destination());
      uncompiledMatchingPattern(instance.uncompiledMatchingPattern());
      return this;
    }

    /**
     * Initializes the value for the {@link FileSet.Entry#type() type} attribute.
     * @param type The value for type 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder type(FileType type) {
      this.type = Objects.requireNonNull(type, "type");
      initBits &= ~INIT_BIT_TYPE;
      return this;
    }

    /**
     * Initializes the value for the {@link FileSet.Entry#destination() destination} attribute.
     * @param destination The value for destination 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder destination(String destination) {
      this.destination = Objects.requireNonNull(destination, "destination");
      initBits &= ~INIT_BIT_DESTINATION;
      return this;
    }

    /**
     * Initializes the value for the {@link FileSet.Entry#uncompiledMatchingPattern() uncompiledMatchingPattern} attribute.
     * @param uncompiledMatchingPattern The value for uncompiledMatchingPattern 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder uncompiledMatchingPattern(UncompiledPattern uncompiledMatchingPattern) {
      this.uncompiledMatchingPattern = Objects.requireNonNull(uncompiledMatchingPattern, "uncompiledMatchingPattern");
      initBits &= ~INIT_BIT_UNCOMPILED_MATCHING_PATTERN;
      return this;
    }

    /**
     * Builds a new {@link ImmutableEntry ImmutableEntry}.
     * @return An immutable instance of Entry
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableEntry build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableEntry(null, type, destination, uncompiledMatchingPattern);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_TYPE) != 0) attributes.add("type");
      if ((initBits & INIT_BIT_DESTINATION) != 0) attributes.add("destination");
      if ((initBits & INIT_BIT_UNCOMPILED_MATCHING_PATTERN) != 0) attributes.add("uncompiledMatchingPattern");
      return "Cannot build Entry, some of required attributes are not set " + attributes;
    }
  }
}
