/*
 * Decompiled with CFR 0.152.
 */
package de.flapdoodle.embed.process.store;

import de.flapdoodle.checks.Preconditions;
import de.flapdoodle.embed.process.archives.ExtractedFileSet;
import de.flapdoodle.embed.process.archives.ImmutableExtractedFileSet;
import de.flapdoodle.embed.process.config.store.FileSet;
import de.flapdoodle.embed.process.hash.Hasher;
import de.flapdoodle.embed.process.store.ExtractedFileSetStore;
import de.flapdoodle.types.Try;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.SeekableByteChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

public class ContentHashExtractedFileSetStore
implements ExtractedFileSetStore {
    private static final int HASH_BUFFER_SIZE = 0x100000;
    private final Path basePath;
    private final Path cachePath;

    public ContentHashExtractedFileSetStore(Path basePath) {
        this.basePath = basePath;
        this.cachePath = basePath.resolve("hashes");
        if (!Files.exists(basePath, new LinkOption[0])) {
            Try.run(() -> Files.createDirectory(basePath, new FileAttribute[0]));
        }
        if (!Files.exists(this.cachePath, new LinkOption[0])) {
            Try.run(() -> Files.createDirectory(this.cachePath, new FileAttribute[0]));
        }
    }

    @Override
    public Optional<ExtractedFileSet> extractedFileSet(Path archive, FileSet fileSet) {
        String hash = ContentHashExtractedFileSetStore.hash(this.cachePath, archive, fileSet);
        Path fileSetBasePath = this.basePath.resolve(hash);
        if (Files.isDirectory(fileSetBasePath, new LinkOption[0])) {
            return (Optional)Try.supplier(() -> Optional.of(ContentHashExtractedFileSetStore.readFileSet(fileSetBasePath, fileSet))).fallbackTo(ex -> Optional.empty()).get();
        }
        return Optional.empty();
    }

    @Override
    public ExtractedFileSet store(Path archive, FileSet fileSet, ExtractedFileSet src) throws IOException {
        String hash = ContentHashExtractedFileSetStore.hash(this.cachePath, archive, fileSet);
        Path fileSetBasePath = this.basePath.resolve(hash);
        Preconditions.checkArgument((!Files.exists(fileSetBasePath, new LinkOption[0]) ? 1 : 0) != 0, (String)"hash collision for %s (hash=%s)", (Object[])new Object[]{archive, hash});
        Files.createDirectory(fileSetBasePath, new FileAttribute[0]);
        return ContentHashExtractedFileSetStore.makeCopyOf(fileSetBasePath, fileSet, src);
    }

    private static ExtractedFileSet makeCopyOf(Path fileSetBasePath, FileSet fileSet, ExtractedFileSet src) throws IOException {
        try {
            Map nameMap = src.libraryFiles().stream().collect(Collectors.toMap(it -> src.baseDir().relativize((Path)it).toString(), Function.identity()));
            ImmutableExtractedFileSet.Builder builder = ExtractedFileSet.builder(fileSetBasePath);
            for (FileSet.Entry entry : fileSet.entries()) {
                Path dest = fileSetBasePath.resolve(entry.destination());
                switch (entry.type()) {
                    case Executable: {
                        if (!Files.exists(dest.getParent(), new LinkOption[0])) {
                            Files.createDirectory(dest.getParent(), new FileAttribute[0]);
                        }
                        Files.copy(src.executable(), dest, StandardCopyOption.COPY_ATTRIBUTES);
                        builder.executable(dest);
                        break;
                    }
                    case Library: {
                        Path srcPath = (Path)nameMap.get(entry.destination());
                        if (srcPath == null) {
                            throw new IOException("could not find entry for " + entry.destination() + " in " + nameMap);
                        }
                        if (!Files.exists(dest.getParent(), new LinkOption[0])) {
                            Files.createDirectory(dest.getParent(), new FileAttribute[0]);
                        }
                        Files.copy(srcPath, dest, StandardCopyOption.COPY_ATTRIBUTES);
                        builder.addLibraryFiles(dest);
                    }
                }
            }
            return builder.build();
        }
        catch (IOException iox) {
            de.flapdoodle.embed.process.io.Files.deleteAll(fileSetBasePath);
            throw iox;
        }
    }

    static String hash(Path cachePath, Path archive, FileSet fileSet) {
        Preconditions.checkArgument((boolean)Files.exists(cachePath, LinkOption.NOFOLLOW_LINKS), (String)"cache does not exsist: %s", (Object[])new Object[]{cachePath});
        Preconditions.checkArgument((boolean)Files.isDirectory(cachePath, LinkOption.NOFOLLOW_LINKS), (String)"cache is not a directory: %s", (Object[])new Object[]{cachePath});
        Optional cacheKey = (Optional)Try.supplier(() -> ContentHashExtractedFileSetStore.cacheHash(archive)).mapException(ex -> new IOException("could not create cache key for " + archive, (Throwable)ex)).onCheckedException(Throwable::printStackTrace).get();
        return ContentHashExtractedFileSetStore.hash(cacheKey.map(cachePath::resolve), archive, fileSet);
    }

    private static String hash(Optional<Path> optCachedHashPath, Path archive, FileSet fileSet) {
        if (optCachedHashPath.isPresent()) {
            Path cachedHashPath = optCachedHashPath.get();
            if (Files.exists(cachedHashPath, new LinkOption[0])) {
                byte[] hashBytes = (byte[])Try.supplier(() -> Files.readAllBytes(cachedHashPath)).mapToUncheckedException(ex -> new RuntimeException("could not read cached key from " + cachedHashPath, (Throwable)ex)).get();
                return new String(hashBytes, StandardCharsets.UTF_8);
            }
            String hash = ContentHashExtractedFileSetStore.hash(archive, fileSet, 0x100000);
            Try.run(() -> Files.write(cachedHashPath, hash.getBytes(StandardCharsets.UTF_8), StandardOpenOption.CREATE_NEW));
            return hash;
        }
        return ContentHashExtractedFileSetStore.hash(archive, fileSet, 0x100000);
    }

    static String cacheHash(Path archive) throws IOException {
        Hasher cacheHasher = Hasher.instance();
        cacheHasher.update(archive.toString());
        cacheHasher.update(Files.getLastModifiedTime(archive, LinkOption.NOFOLLOW_LINKS).toString());
        return cacheHasher.hashAsString();
    }

    @Deprecated
    static String hash(Path archive, FileSet fileSet) {
        Hasher digest = Hasher.instance();
        fileSet.entries().forEach(entry -> {
            digest.update(entry.type().name().getBytes(StandardCharsets.UTF_8));
            digest.update(entry.destination().getBytes(StandardCharsets.UTF_8));
            digest.update(entry.matchingPattern().toString().getBytes(StandardCharsets.UTF_8));
        });
        digest.update("--".getBytes(StandardCharsets.UTF_8));
        digest.update((byte[])Try.get(() -> Files.readAllBytes(archive)));
        return digest.hashAsString();
    }

    static String hash(Path archive, FileSet fileSet, int blocksize) {
        Hasher digest = Hasher.instance();
        fileSet.entries().forEach(entry -> {
            digest.update(entry.type().name().getBytes(StandardCharsets.UTF_8));
            digest.update(entry.destination().getBytes(StandardCharsets.UTF_8));
            digest.update(entry.matchingPattern().toString().getBytes(StandardCharsets.UTF_8));
        });
        digest.update("--".getBytes(StandardCharsets.UTF_8));
        Try.run(() -> {
            ByteBuffer buffer = ByteBuffer.allocate(blocksize);
            try (SeekableByteChannel channel = Files.newByteChannel(archive, new OpenOption[0]);){
                while (channel.read(buffer) > 0) {
                    buffer.flip();
                    digest.update(buffer);
                    buffer.clear();
                }
            }
        });
        return digest.hashAsString();
    }

    private static ExtractedFileSet readFileSet(Path fileSetBasePath, FileSet fileSet) {
        ImmutableExtractedFileSet.Builder builder = ExtractedFileSet.builder(fileSetBasePath);
        fileSet.entries().forEach(entry -> {
            Path entryPath = fileSetBasePath.resolve(entry.destination());
            Preconditions.checkArgument((boolean)Files.exists(entryPath, new LinkOption[0]), (String)"could not find matching file: %s", (Object[])new Object[]{entryPath});
            switch (entry.type()) {
                case Executable: {
                    builder.executable(entryPath);
                    break;
                }
                case Library: {
                    builder.addLibraryFiles(entryPath);
                }
            }
        });
        return builder.build();
    }
}

