package de.flapdoodle.embed.mongo.spring.autoconfigure;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import org.bson.Document;

/**
 * Immutable implementation of {@link MongoClientAction}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableMongoClientAction.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableMongoClientAction
    extends MongoClientAction {
  private final MongoClientAction.Credentials credentials;
  private final MongoClientAction.Action action;
  private final Consumer<Document> onResult;
  private final Consumer<RuntimeException> onError;

  private ImmutableMongoClientAction(ImmutableMongoClientAction.Builder builder) {
    this.credentials = builder.credentials;
    this.action = builder.action;
    if (builder.onResult != null) {
      initShim.onResult(builder.onResult);
    }
    if (builder.onError != null) {
      initShim.onError(builder.onError);
    }
    this.onResult = initShim.onResult();
    this.onError = initShim.onError();
    this.initShim = null;
  }

  private ImmutableMongoClientAction(
      MongoClientAction.Credentials credentials,
      MongoClientAction.Action action,
      Consumer<Document> onResult,
      Consumer<RuntimeException> onError) {
    this.credentials = credentials;
    this.action = action;
    this.onResult = onResult;
    this.onError = onError;
    this.initShim = null;
  }

  private static final byte STAGE_INITIALIZING = -1;
  private static final byte STAGE_UNINITIALIZED = 0;
  private static final byte STAGE_INITIALIZED = 1;
  private transient volatile InitShim initShim = new InitShim();

  private final class InitShim {
    private byte onResultBuildStage = STAGE_UNINITIALIZED;
    private Consumer<Document> onResult;

    Consumer<Document> onResult() {
      if (onResultBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (onResultBuildStage == STAGE_UNINITIALIZED) {
        onResultBuildStage = STAGE_INITIALIZING;
        this.onResult = Objects.requireNonNull(ImmutableMongoClientAction.super.onResult(), "onResult");
        onResultBuildStage = STAGE_INITIALIZED;
      }
      return this.onResult;
    }

    void onResult(Consumer<Document> onResult) {
      this.onResult = onResult;
      onResultBuildStage = STAGE_INITIALIZED;
    }

    private byte onErrorBuildStage = STAGE_UNINITIALIZED;
    private Consumer<RuntimeException> onError;

    Consumer<RuntimeException> onError() {
      if (onErrorBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (onErrorBuildStage == STAGE_UNINITIALIZED) {
        onErrorBuildStage = STAGE_INITIALIZING;
        this.onError = Objects.requireNonNull(ImmutableMongoClientAction.super.onError(), "onError");
        onErrorBuildStage = STAGE_INITIALIZED;
      }
      return this.onError;
    }

    void onError(Consumer<RuntimeException> onError) {
      this.onError = onError;
      onErrorBuildStage = STAGE_INITIALIZED;
    }

    private String formatInitCycleMessage() {
      List<String> attributes = new ArrayList<>();
      if (onResultBuildStage == STAGE_INITIALIZING) attributes.add("onResult");
      if (onErrorBuildStage == STAGE_INITIALIZING) attributes.add("onError");
      return "Cannot build MongoClientAction, attribute initializers form cycle " + attributes;
    }
  }

  /**
   * @return The value of the {@code credentials} attribute
   */
  @Override
  public Optional<MongoClientAction.Credentials> credentials() {
    return Optional.ofNullable(credentials);
  }

  /**
   * @return The value of the {@code action} attribute
   */
  @Override
  public MongoClientAction.Action action() {
    return action;
  }

  /**
   * @return The value of the {@code onResult} attribute
   */
  @Override
  public Consumer<Document> onResult() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.onResult()
        : this.onResult;
  }

  /**
   * @return The value of the {@code onError} attribute
   */
  @Override
  public Consumer<RuntimeException> onError() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.onError()
        : this.onError;
  }

  /**
   * Copy the current immutable object by setting a <i>present</i> value for the optional {@link MongoClientAction#credentials() credentials} attribute.
   * @param value The value for credentials
   * @return A modified copy of {@code this} object
   */
  public final ImmutableMongoClientAction withCredentials(MongoClientAction.Credentials value) {
    MongoClientAction.Credentials newValue = Objects.requireNonNull(value, "credentials");
    if (this.credentials == newValue) return this;
    return new ImmutableMongoClientAction(newValue, this.action, this.onResult, this.onError);
  }

  /**
   * Copy the current immutable object by setting an optional value for the {@link MongoClientAction#credentials() credentials} attribute.
   * A shallow reference equality check is used on unboxed optional value to prevent copying of the same value by returning {@code this}.
   * @param optional A value for credentials
   * @return A modified copy of {@code this} object
   */
  @SuppressWarnings("unchecked") // safe covariant cast
  public final ImmutableMongoClientAction withCredentials(Optional<? extends MongoClientAction.Credentials> optional) {
    MongoClientAction.Credentials value = optional.orElse(null);
    if (this.credentials == value) return this;
    return new ImmutableMongoClientAction(value, this.action, this.onResult, this.onError);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link MongoClientAction#action() action} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for action
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableMongoClientAction withAction(MongoClientAction.Action value) {
    if (this.action == value) return this;
    MongoClientAction.Action newValue = Objects.requireNonNull(value, "action");
    return new ImmutableMongoClientAction(this.credentials, newValue, this.onResult, this.onError);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link MongoClientAction#onResult() onResult} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for onResult
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableMongoClientAction withOnResult(Consumer<Document> value) {
    if (this.onResult == value) return this;
    Consumer<Document> newValue = Objects.requireNonNull(value, "onResult");
    return new ImmutableMongoClientAction(this.credentials, this.action, newValue, this.onError);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link MongoClientAction#onError() onError} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for onError
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableMongoClientAction withOnError(Consumer<RuntimeException> value) {
    if (this.onError == value) return this;
    Consumer<RuntimeException> newValue = Objects.requireNonNull(value, "onError");
    return new ImmutableMongoClientAction(this.credentials, this.action, this.onResult, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableMongoClientAction} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableMongoClientAction
        && equalTo(0, (ImmutableMongoClientAction) another);
  }

  private boolean equalTo(int synthetic, ImmutableMongoClientAction another) {
    return Objects.equals(credentials, another.credentials)
        && action.equals(another.action)
        && onResult.equals(another.onResult)
        && onError.equals(another.onError);
  }

  /**
   * Computes a hash code from attributes: {@code credentials}, {@code action}, {@code onResult}, {@code onError}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(credentials);
    h += (h << 5) + action.hashCode();
    h += (h << 5) + onResult.hashCode();
    h += (h << 5) + onError.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code MongoClientAction} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    StringBuilder builder = new StringBuilder("MongoClientAction{");
    if (credentials != null) {
      builder.append("credentials=").append(credentials);
    }
    if (builder.length() > 18) builder.append(", ");
    builder.append("action=").append(action);
    builder.append(", ");
    builder.append("onResult=").append(onResult);
    builder.append(", ");
    builder.append("onError=").append(onError);
    return builder.append("}").toString();
  }

  /**
   * Creates an immutable copy of a {@link MongoClientAction} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable MongoClientAction instance
   */
  public static ImmutableMongoClientAction copyOf(MongoClientAction instance) {
    if (instance instanceof ImmutableMongoClientAction) {
      return (ImmutableMongoClientAction) instance;
    }
    return ImmutableMongoClientAction.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableMongoClientAction ImmutableMongoClientAction}.
   * <pre>
   * ImmutableMongoClientAction.builder()
   *    .credentials(de.flapdoodle.embed.mongo.spring.autoconfigure.MongoClientAction.Credentials) // optional {@link MongoClientAction#credentials() credentials}
   *    .action(de.flapdoodle.embed.mongo.spring.autoconfigure.MongoClientAction.Action) // required {@link MongoClientAction#action() action}
   *    .onResult(function.Consumer&amp;lt;org.bson.Document&amp;gt;) // optional {@link MongoClientAction#onResult() onResult}
   *    .onError(function.Consumer&amp;lt;RuntimeException&amp;gt;) // optional {@link MongoClientAction#onError() onError}
   *    .build();
   * </pre>
   * @return A new ImmutableMongoClientAction builder
   */
  public static ImmutableMongoClientAction.Builder builder() {
    return new ImmutableMongoClientAction.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableMongoClientAction ImmutableMongoClientAction}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private static final long INIT_BIT_ACTION = 0x1L;
    private long initBits = 0x1L;

    private MongoClientAction.Credentials credentials;
    private MongoClientAction.Action action;
    private Consumer<Document> onResult;
    private Consumer<RuntimeException> onError;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code MongoClientAction} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(MongoClientAction instance) {
      Objects.requireNonNull(instance, "instance");
      Optional<MongoClientAction.Credentials> credentialsOptional = instance.credentials();
      if (credentialsOptional.isPresent()) {
        credentials(credentialsOptional);
      }
      this.action(instance.action());
      this.onResult(instance.onResult());
      this.onError(instance.onError());
      return this;
    }

    /**
     * Initializes the optional value {@link MongoClientAction#credentials() credentials} to credentials.
     * @param credentials The value for credentials
     * @return {@code this} builder for chained invocation
     */
    public final Builder credentials(MongoClientAction.Credentials credentials) {
      this.credentials = Objects.requireNonNull(credentials, "credentials");
      return this;
    }

    /**
     * Initializes the optional value {@link MongoClientAction#credentials() credentials} to credentials.
     * @param credentials The value for credentials
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder credentials(Optional<? extends MongoClientAction.Credentials> credentials) {
      this.credentials = credentials.orElse(null);
      return this;
    }

    /**
     * Initializes the value for the {@link MongoClientAction#action() action} attribute.
     * @param action The value for action 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder action(MongoClientAction.Action action) {
      this.action = Objects.requireNonNull(action, "action");
      initBits &= ~INIT_BIT_ACTION;
      return this;
    }

    /**
     * Initializes the value for the {@link MongoClientAction#onResult() onResult} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link MongoClientAction#onResult() onResult}.</em>
     * @param onResult The value for onResult 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder onResult(Consumer<Document> onResult) {
      this.onResult = Objects.requireNonNull(onResult, "onResult");
      return this;
    }

    /**
     * Initializes the value for the {@link MongoClientAction#onError() onError} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link MongoClientAction#onError() onError}.</em>
     * @param onError The value for onError 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder onError(Consumer<RuntimeException> onError) {
      this.onError = Objects.requireNonNull(onError, "onError");
      return this;
    }

    /**
     * Builds a new {@link ImmutableMongoClientAction ImmutableMongoClientAction}.
     * @return An immutable instance of MongoClientAction
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableMongoClientAction build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableMongoClientAction(this);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_ACTION) != 0) attributes.add("action");
      return "Cannot build MongoClientAction, some of required attributes are not set " + attributes;
    }
  }
}
