/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.maven.plugin;

import ca.uhn.fhir.context.FhirContext;
import ca.uhn.fhir.context.FhirVersionEnum;
import com.google.common.base.Strings;
import de.fhlintstone.process.IProcessor;
import de.fhlintstone.process.ProcessException;
import de.fhlintstone.process.config.NestedClassConfiguration;
import de.fhlintstone.process.config.PackageSourceConfiguration;
import de.fhlintstone.process.config.ProcessConfiguration;
import de.fhlintstone.process.config.StructureClassInstantiation;
import de.fhlintstone.process.config.StructureDefinitionClassConfiguration;
import de.fhlintstone.process.config.ValueSetEnumConfiguration;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import javax.inject.Inject;
import lombok.extern.slf4j.XSlf4j;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;

/**
 * Goal to generate the source code according to the configuration.
 */
@Mojo(name = "generate-code", defaultPhase = LifecyclePhase.GENERATE_SOURCES)
@XSlf4j
public class CodeGeneratorMojo extends AbstractMojo {

    private final IProcessor processor;

    /**
     * The FHIR base version to use. Among other things, this controls the
     * {@link FhirContext} instantiation.
     */
    @Parameter(name = "fhirVersion", required = true)
    private FhirVersionEnum fhirVersion;

    /**
     * The output path to use for code generation.
     */
    @Parameter(name = "outputPath", required = true)
    private String outputPath;

    /**
     * The default namespace to use for code generation.
     */
    @Parameter(name = "defaultNamespace", required = false)
    private String defaultNamespace;

    /**
     * The list of package sources to use for generation.
     */
    @Parameter(name = "packageSources", required = true)
    private final List<PackageSource> packageSources = new ArrayList<>();

    /**
     * The list of ValueSet enum generator rules to use.
     */
    @Parameter(name = "valueSetEnums", required = false)
    private final List<ValueSetEnum> valueSetEnums = new ArrayList<>();

    /**
     * The list of StructureDefinition class generator rules to use.
     */
    @Parameter(name = "structureDefinitionClasses", required = false)
    private final List<StructureDefinitionClass> structureDefinitionClasses = new ArrayList<>();

    /**
     * Constructor for dependency injection.
     *
     * @param processor the {@link IProcessor} to use
     */
    @Inject
    public CodeGeneratorMojo(IProcessor processor) {
        super();
        this.processor = processor;
    }

    @Override
    public void execute() throws MojoExecutionException {
        logger.entry();
        final var config = convertToCoreConfiguration();
        try {
            this.processor.execute(config);
        } catch (final ProcessException e) {
            throw new MojoExecutionException(e.getMessage(), e);
        }
        logger.exit();
    }

    /**
     * Converts the configuration of this Mojo to the config object used by the Fhlintstone core.
     * @return the converted configuration
     */
    public ProcessConfiguration convertToCoreConfiguration() {
        logger.entry();
        final var packageSourceConfigurations = new ArrayList<PackageSourceConfiguration>();
        for (final var packageSource : this.packageSources) {
            packageSourceConfigurations.add(PackageSourceConfiguration.builder()
                    .withSourceFile(Optional.of(packageSource.getSourceFile()))
                    .build());
        }

        final var valueSetEnumConfigurations = new ArrayList<ValueSetEnumConfiguration>();
        for (final var valueSetEnum : this.valueSetEnums) {
            valueSetEnumConfigurations.add(ValueSetEnumConfiguration.builder()
                    .withValueSet(valueSetEnum.getValueSet())
                    .withTargetNamespace(Optional.ofNullable(valueSetEnum.getTargetNamespace()))
                    .withEnumName(valueSetEnum.getEnumName())
                    .build());
        }
        final var structureDefinitionClassConfigurations = new ArrayList<StructureDefinitionClassConfiguration>();
        for (final var structureDefinitionClass : this.structureDefinitionClasses) {
            final var classBuilder = StructureDefinitionClassConfiguration.builder()
                    .withStructureDefinition(structureDefinitionClass.getStructureDefinition())
                    .withTargetNamespace(Optional.ofNullable(structureDefinitionClass.getTargetNamespace()))
                    .withClassName(structureDefinitionClass.getClassName())
                    .withInstantiation(convertInstantiationMode(structureDefinitionClass.getInstantiation()))
                    .withUseVersion(structureDefinitionClass.useVersion());
            for (final var nestedClass : structureDefinitionClass.getNestedClasses()) {
                classBuilder.withNestedClass(NestedClassConfiguration.builder()
                        .withElementId(nestedClass.getElementId())
                        .withClassName(nestedClass.getClassName())
                        .withSuperClass(nestedClass.getSuperClass())
                        .withInstantiation(convertInstantiationMode(nestedClass.getInstantiation()))
                        .build());
            }
            structureDefinitionClassConfigurations.add(classBuilder.build());
        }
        final var config = ProcessConfiguration.builder()
                .withFhirVersion(this.fhirVersion)
                .withOutputPath(this.outputPath)
                .withDefaultNamespace(Optional.ofNullable(this.defaultNamespace))
                .withPackageSources(packageSourceConfigurations)
                .withValueSetEnums(valueSetEnumConfigurations)
                .withStructureDefinitionClasses(structureDefinitionClassConfigurations)
                .build();
        return logger.exit(config);
    }

    private StructureClassInstantiation convertInstantiationMode(String mode) {
        logger.entry(mode);
        final var instantiationEnum = Strings.isNullOrEmpty(mode)
                ? StructureClassInstantiation.DEFAULT
                : StructureClassInstantiation.fromString(mode);
        return logger.exit(instantiationEnum);
    }
}
