/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.process.config;

import com.google.common.base.Strings;

/**
 * Describes how the generated class can be instantiated.
 */
public enum StructureClassInstantiation {
    /**
     * Make the class abstract to prevent instantiation.
     */
    ABSTRACT,

    /**
     * Generate a default constructor with no further provisions.
     */
    DEFAULT,

    /**
     * Generate a mandatory builder.
     */
    BUILDER;

    /**
     * Converts a case-insensitive string to an enum value.
     * @param str the string to convert
     * @return the corresponding value
     * @throws IllegalArgumentException on conversion errors
     */
    public static StructureClassInstantiation fromString(String str) throws IllegalArgumentException {
        if (Strings.isNullOrEmpty(str)) {
            throw new IllegalArgumentException("Unable to convert empty string");
        }
        if (str.equalsIgnoreCase("abstract")) {
            return StructureClassInstantiation.ABSTRACT;
        } else if (str.equalsIgnoreCase("default")) {
            return StructureClassInstantiation.DEFAULT;
        } else if (str.equalsIgnoreCase("builder")) {
            return StructureClassInstantiation.BUILDER;
        } else {
            throw new IllegalArgumentException(String.format(
                    "Invalid instantiation setting '%s'. Valid values are abstract, default, builder.", str));
        }
    }
}
