/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.process.config;

import ca.uhn.fhir.context.FhirVersionEnum;
import com.google.common.collect.ImmutableList;
import java.net.URI;
import java.util.Optional;
import lombok.Builder;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.Singular;
import lombok.extern.slf4j.XSlf4j;

/**
 * This class describes the configuration used to generate the source code.
 */
@EqualsAndHashCode
@Builder(setterPrefix = "with")
@XSlf4j
public final class ProcessConfiguration {

    /**
     * The FHIR version to use.
     */
    @Getter
    private final FhirVersionEnum fhirVersion;

    /**
     * The output path to use.
     */
    @Getter
    private final String outputPath;

    /**
     * The default target namespace, if set.
     */
    @Getter
    @Builder.Default
    private final Optional<String> defaultNamespace = Optional.empty();

    /**
     * The configured package sources.
     */
    @Getter
    @Singular
    private final ImmutableList<PackageSourceConfiguration> packageSources;

    /**
     * The configured enums to generate for ValueSets.
     */
    @Getter
    @Singular
    private final ImmutableList<ValueSetEnumConfiguration> valueSetEnums;

    /**
     * The configured classes to generate for StructureDefinitions.
     */
    @Getter
    @Singular
    private final ImmutableList<StructureDefinitionClassConfiguration> structureDefinitionClasses;

    /**
     * Attempts to find the configuration for a StructureDefinition identified by its URI.
     * @param structureDefinitionURI the URI to search for
     * @return the {@link StructureDefinitionClassConfiguration}, if an entry for the StructureDefinition exists
     */
    public Optional<StructureDefinitionClassConfiguration> getStructureDefinition(URI structureDefinitionURI) {
        logger.entry(structureDefinitionURI);
        return logger.exit(this.structureDefinitionClasses.stream()
                .filter(c -> c.getStructureDefinition().equals(structureDefinitionURI))
                .findAny());
    }
}
