/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.process.config;

import com.google.common.base.Strings;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import com.palantir.javapoet.ClassName;
import com.palantir.javapoet.ParameterizedTypeName;
import com.palantir.javapoet.TypeName;
import java.util.Optional;
import java.util.regex.Pattern;
import lombok.Builder;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.extern.slf4j.XSlf4j;

/**
 * This class describes a nested class contained in a {@link StructureDefinitionClassConfiguration} to
 * represent a component or an anonymous local type.
 */
@EqualsAndHashCode
@Builder(setterPrefix = "with")
@XSlf4j
public final class NestedClassConfiguration {

    /**
     * The full ID of the element for which the nested class is generated (e.g. Patient.name).
     */
    @Getter
    private final String elementId;

    /**
     * The name of the nested class to generate.
     */
    @Getter
    private final String className;

    /**
     * The name of the superclass the nested class is derived from.
     */
    @Getter
    private final String superClass;

    /**
     * The instantiation mode to use.
     */
    @Getter
    private final StructureClassInstantiation instantiation;

    @SuppressWarnings("java:S4738") // Java supplier does not support memoization
    private final Supplier<TypeName> superTypeNameSupplier = Suppliers.memoize(() -> determineSuperTypeName());

    private TypeName determineSuperTypeName() throws IllegalStateException {
        logger.entry(this.superClass);
        final var pattern = Pattern.compile("^([\\w\\.]+)(?:<([\\w\\.]+)>)?$");
        final var matcher = pattern.matcher(this.superClass);
        if (matcher.find()) {
            // group 2 is the non-capturing group... whatever *sigh*
            final var baseName = matcher.group(1);
            final var paramName = matcher.group(2);
            if (Strings.isNullOrEmpty(paramName)) {
                final var parsedClassName = matcher.group(1);
                logger.debug("Parent type is not parameterized, class name is {}", parsedClassName);
                final var parentType = ClassName.bestGuess(parsedClassName);
                return logger.exit(parentType);
            } else {
                logger.debug(
                        "Parent type is parameterized, base class name is {}, parameter is {}", baseName, paramName);
                final var baseType = ClassName.bestGuess(baseName);
                final var paramType = ClassName.bestGuess(paramName);
                final var parameterizedType = ParameterizedTypeName.get(baseType, paramType);
                return logger.exit(parameterizedType);
            }
        }
        throw logger.throwing(new IllegalStateException(
                String.format("Configured superclass name %s appears to be invalid", this.superClass)));
    }

    /**
     * Determine the name of the superclass of the nested type as a {@link TypeName}.
     *
     * This might be a parameterized type like
     * <code>org.hl7.fhir.r4.model.Enumeration&lt;org.hl7.fhir.r4.model.Enumerations.AdministrativeGender&gt;</code>,
     * and since the JavaPoet type system can't bestGuess it's way out of this, we have to handle this ourselves.
     *
     * @return the type name to use as the parent type, or an empty value if no superclass is set
     */
    public Optional<TypeName> getSuperTypeName() {
        return Strings.isNullOrEmpty(this.superClass)
                ? Optional.empty()
                : Optional.of(this.superTypeNameSupplier.get());
    }
}
