/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.process;

import com.google.common.collect.ImmutableSet;
import de.fhlintstone.accessors.IAccessorProvider;
import de.fhlintstone.accessors.UnsupportedFHIRVersionException;
import de.fhlintstone.fhir.FhirException;
import de.fhlintstone.fhir.IStructureDefinitionIntrospector;
import de.fhlintstone.fhir.dependencies.DependencyException;
import de.fhlintstone.fhir.dependencies.IDependency.Origin;
import de.fhlintstone.fhir.dependencies.IDependencyGraph;
import de.fhlintstone.fhir.dependencies.IDependencyGraphBuilder;
import de.fhlintstone.fhir.dependencies.IDependencyNode;
import de.fhlintstone.generator.GeneratorException;
import de.fhlintstone.generator.IGeneratedTypeNameRegistry;
import de.fhlintstone.generator.structuredefinition.IStructureDefinitionGenerator;
import de.fhlintstone.generator.valueset.IValueSetGenerator;
import de.fhlintstone.packages.IPackageRegistry;
import de.fhlintstone.packages.PackageFileException;
import de.fhlintstone.process.config.ProcessConfiguration;
import java.io.File;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Optional;
import javax.inject.Inject;
import javax.inject.Named;
import lombok.extern.slf4j.XSlf4j;
import org.hl7.fhir.instance.model.api.IBaseResource;

/**
 * Default implementation of {@link IProcessor}.
 */
@Named
@XSlf4j
public class Processor implements IProcessor {

    private final IContextProvider contextProvider;
    private final IAccessorProvider accessorProvider;
    private final IPackageRegistry packageRegistry;
    private final IStructureDefinitionIntrospector structureDefinitionIntrospector;
    private final IDependencyGraphBuilder dependencyGraphBuilder;
    private final IGeneratedTypeNameRegistry generatedTypeNameRegistry;
    private final IValueSetGenerator valueSetGenerator;
    private final IStructureDefinitionGenerator structureDefinitionGenerator;

    /**
     * Constructor for dependency injection.
     *
     * @param contextProvider              the {@link IContextProvider} to use
     * @param accessorProvider the {@link IAccessorProvider} to use
     * @param packageRegistry              the {@link IPackageRegistry} to use
     * @param structureDefinitionIntrospector the {@link IStructureDefinitionIntrospector} to use
     * @param dependencyGraphBuilder       the {@link IDependencyGraphBuilder} to
     *                                     use
     * @param generatedTypeNameRegistry    the {@link IGeneratedTypeNameRegistry} to
     *                                     use
     * @param valueSetGenerator            the {@link IValueSetGenerator} to use
     * @param structureDefinitionGenerator the {@link IStructureDefinitionGenerator}
     *                                     to use
     */
    @Inject
    public Processor(
            IContextProvider contextProvider,
            IAccessorProvider accessorProvider,
            IPackageRegistry packageRegistry,
            IStructureDefinitionIntrospector structureDefinitionIntrospector,
            IDependencyGraphBuilder dependencyGraphBuilder,
            IGeneratedTypeNameRegistry generatedTypeNameRegistry,
            IValueSetGenerator valueSetGenerator,
            IStructureDefinitionGenerator structureDefinitionGenerator) {
        this.contextProvider = contextProvider;
        this.accessorProvider = accessorProvider;
        this.packageRegistry = packageRegistry;
        this.structureDefinitionIntrospector = structureDefinitionIntrospector;
        this.dependencyGraphBuilder = dependencyGraphBuilder;
        this.generatedTypeNameRegistry = generatedTypeNameRegistry;
        this.valueSetGenerator = valueSetGenerator;
        this.structureDefinitionGenerator = structureDefinitionGenerator;
    }

    @Override
    public void execute(ProcessConfiguration configuration) throws ProcessException {
        logger.entry(configuration);
        configureFhirVersion(configuration);
        registerPackages(configuration);
        generateTypeNames(configuration);
        final var dependencyGraph = checkDependencies(configuration);
        var generatorSuccess = true;
        generatorSuccess &= this.valueSetGenerator.generate(configuration, dependencyGraph);
        generatorSuccess &= this.structureDefinitionGenerator.generate(configuration, dependencyGraph);
        if (!generatorSuccess) {
            throw logger.throwing(new ProcessException("Error generating code, check logs for more information."));
        }
        logger.exit();
    }

    /**
     * Configures the framework to use the correct FHIR version.
     *
     * @param configuration
     * @throws ProcessException
     */
    private void configureFhirVersion(ProcessConfiguration configuration) throws ProcessException {
        logger.entry(configuration);
        final var fhirVersion = configuration.getFhirVersion();
        if (fhirVersion == null) {
            throw logger.throwing(new ProcessException("FHIR version is not configured"));
        }
        switch (fhirVersion) {
            case R4, R4B, R5:
                logger.debug("Using FHIR version {}", fhirVersion);
                this.contextProvider.setFhirVersion(fhirVersion);
                break;
            default:
                throw logger.throwing(new UnsupportedFHIRVersionException(fhirVersion));
        }
        logger.exit();
    }

    /**
     * Ensures that the FHIR packages are loaded and accessible as configured.
     *
     * @param configuration
     * @throws ProcessException
     */
    private void registerPackages(ProcessConfiguration configuration) throws ProcessException {
        logger.entry(configuration);
        final var packageSources = configuration.getPackageSources();
        if (packageSources.isEmpty()) {
            throw logger.throwing(new ProcessException("No source packages have been specified"));
        }
        this.packageRegistry.clear();
        for (final var packageSource : packageSources) {
            final Optional<File> sourceFile = packageSource.getSourceFile();
            if (sourceFile.isPresent()) {
                try {
                    this.packageRegistry.register(sourceFile.get());
                } catch (final PackageFileException e) {
                    throw logger.throwing(new ProcessException("Unable to register a source package", e));
                }
            } else {
                throw logger.throwing(new ProcessException("Unable to register a source package"));
            }
        }
        final var unmetDependencies = this.packageRegistry.getUnmetPackageDependencies();
        if (!unmetDependencies.isEmpty()) {
            // TODO #22 introduce a configuration for ignored dependencies
            logger.warn(
                    "The following package-level dependencies can not be resolved: {}",
                    String.join(", ", unmetDependencies));
        }
        logger.exit();
    }

    /**
     * Ensures that the {@link IGeneratedTypeNameRegistry} has the current
     * configuration set and that the name settings are all valid.
     *
     * @param configuration
     * @throws ProcessException
     */
    private void generateTypeNames(ProcessConfiguration configuration) throws ProcessException {
        try {
            this.generatedTypeNameRegistry.setConfiguration(configuration);
        } catch (final GeneratorException e) {
            throw logger.throwing(
                    new ProcessException("The configuration contains invalid namespace or target name settings", e));
        }
    }

    /**
     * Checks whether all dependencies are included in the configuration.
     *
     * @param configuration
     * @throws ProcessException
     */
    private IDependencyGraph checkDependencies(ProcessConfiguration configuration) throws ProcessException {
        logger.entry(configuration);

        final var relevantOrigins = ImmutableSet.of(
                Origin.BASE_DEFINITION, Origin.BINDING_VALUE_SET, Origin.TYPE_CODE, Origin.TYPE_PROFILE
                // Origin.TYPE_TARGET_PROFILE,
                // Origin.COMPOSE_CODE_SYSTEM,
                // Origin.COMPOSE_VALUE_SET,
                // Origin.CODE_SYSTEM_SUPPLEMENTS,
                // Origin.CODE_SYSTEM_VALUE_SET;
                );

        final var configuredResources = collectConfiguredResources(configuration);
        final var dependencyGraph = buildDependencyGraph(configuration);
        final var missingDependencies = new ArrayList<IDependencyNode>();
        final var nodesWithoutFrameworkImplementation = dependencyGraph.getNodes().stream()
                .filter(n -> n.getFrameworkType().isEmpty())
                .toList();
        for (final var node : nodesWithoutFrameworkImplementation) {
            final URI resourceURI = node.getResourceURI();
            if (!configuredResources.contains(resourceURI)) {
                final var origins = dependencyGraph.getDependentOrigins(node);
                if (!Collections.disjoint(origins, relevantOrigins)) {
                    missingDependencies.add(node);
                }
            }
        }
        if (!missingDependencies.isEmpty()) {
            missingDependencies.sort((o1, o2) ->
                    o1.getResourceURI().toString().compareTo(o2.getResource().toString()));
            logger.warn(
                    "The following {} resource(s) are a dependency of the configured resources, but not covered by the configuration or the HAPI base classes.",
                    missingDependencies.size());
            for (final var node : missingDependencies) {
                logger.warn("  {}", node.getResourceURI());
                final var deps = new ArrayList<>(dependencyGraph.getDependents(node));
                deps.sort((o1, o2) -> {
                    final var k1 =
                            o1.getOrigin().toString() + "|" + o1.getDependent().getResourceURI();
                    final var k2 =
                            o2.getOrigin().toString() + "|" + o2.getDependent().getResourceURI();
                    return k1.compareTo(k2);
                });
                for (final var dep : deps) {
                    logger.warn(
                            "    used as {} by {}",
                            dep.getOrigin(),
                            dep.getDependent().getResourceURI());
                }
            }
        }
        return logger.exit(dependencyGraph);
    }

    /**
     * Determines the URIs of all configured resources.
     *
     * @param configuration
     * @return
     */
    private ImmutableSet<URI> collectConfiguredResources(ProcessConfiguration configuration) {
        logger.entry(configuration);
        final var result = new HashSet<URI>();
        configuration.getValueSetEnums().stream().map(c -> c.getValueSet()).forEach(result::add);
        configuration.getStructureDefinitionClasses().stream()
                .map(c -> c.getStructureDefinition())
                .forEach(result::add);
        return logger.exit(ImmutableSet.copyOf(result));
    }

    /**
     * Creates a dependency graph of the required objects.
     *
     * @param configuration
     * @return
     * @throws ProcessException
     */
    private IDependencyGraph buildDependencyGraph(ProcessConfiguration configuration) throws ProcessException {
        logger.entry(configuration);
        for (final var valueSetConfig : configuration.getValueSetEnums()) {
            addResourceToDependencyGraph(valueSetConfig.getValueSet());
        }
        for (final var structureDefinitionConfig : configuration.getStructureDefinitionClasses()) {
            final var resource = addResourceToDependencyGraph(structureDefinitionConfig.getStructureDefinition());
            for (final var nestedConfig : structureDefinitionConfig.getNestedClasses()) {
                final var accessor = this.accessorProvider.provideStructureDefinitionAccessor(resource);
                try {
                    final var typeSpecifications = this.structureDefinitionIntrospector.getElementTypeSpecifications(
                            accessor, nestedConfig.getElementId());
                    for (final var typeSpecification : typeSpecifications) {
                        for (final var profile : typeSpecification.getProfiles()) {
                            addResourceToDependencyGraph(profile.canonical());
                        }
                        addResourceToDependencyGraph(
                                typeSpecification.getTypeCode().canonical());
                    }
                } catch (final FhirException e) {
                    throw new ProcessException(e.getMessage(), e);
                }
            }
        }
        try {
            return logger.exit(this.dependencyGraphBuilder.build());
        } catch (final DependencyException e) {
            throw logger.throwing(new ProcessException("Error building dependency graph", e));
        }
    }

    /**
     * Adds a resource (by URI) to the dependency graph builder.
     *
     * @param resourceURI
     * @return
     * @throws ProcessException
     */
    private IBaseResource addResourceToDependencyGraph(final URI resourceURI) throws ProcessException {
        logger.entry(resourceURI);
        final var resource = this.packageRegistry.getUniqueResource(resourceURI);
        if (resource.isEmpty()) {
            throw logger.throwing(new ProcessException(String.format("Unable to load Resource %s", resourceURI)));
        }
        try {
            if (!this.dependencyGraphBuilder.containsResource(resourceURI)) {
                this.dependencyGraphBuilder.addResource(resource.get());
            }
        } catch (final DependencyException e) {
            throw logger.throwing(new ProcessException(
                    String.format("Unable to add Resource %s to the dependency graph builder", resourceURI), e));
        }
        return logger.exit(resource.get());
    }
}
