/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.process;

import ca.uhn.fhir.context.BaseRuntimeElementDefinition;
import ca.uhn.fhir.context.FhirContext;
import ca.uhn.fhir.context.FhirVersionEnum;
import ca.uhn.fhir.context.RuntimeResourceDefinition;
import ca.uhn.fhir.parser.IParser;
import java.util.Optional;
import org.hl7.fhir.instance.model.api.IBase;
import org.hl7.fhir.instance.model.api.IBaseResource;

/**
 * Provides access to the {@link FhirContext}. The context is created on first
 * request according to the FHIR version that has to be set beforehand.
 */
public interface IContextProvider {

    /**
     * Returns the FHIR version, if set.
     *
     * @return the FHIR version, if set
     */
    Optional<FhirVersionEnum> getFhirVersion();

    /**
     * Changes the FHIR version to use. This method may be called any number of
     * times before the {@link FhirContext} is obtained for the first time. Any
     * attempt to change the FHIR version after the {@link FhirContext} has been
     * created will yield an {@link IllegalStateException}
     *
     * @param fhirVersion
     * @throws IllegalStateException if a {@link FhirContext} with a different
     *                               version has already been created.
     */
    void setFhirVersion(final FhirVersionEnum fhirVersion) throws IllegalStateException;

    /**
     * Create and return a new JSON parser.
     *
     * @return a new JSON parser
     * @see FhirContext#newJsonParser()
     */
    IParser newJsonParser();

    /**
     * Returns the scanned runtime model for the given type.
     *
     * @param theElementName The name of the element to load the element definition for
     * @return the model, if any
     * @see FhirContext#getElementDefinition(String)
     */
    @SuppressWarnings("java:S1452") // wrapped HAPI method uses the same generic type
    Optional<BaseRuntimeElementDefinition<?>> getElementDefinition(final String theElementName);

    /**
     * Returns the scanned runtime model for the given type.
     *
     * @param theElementType The type of the element to load the element definition for
     * @return the model, if any
     * @see FhirContext#getElementDefinition(Class)
     */
    @SuppressWarnings("java:S1452") // wrapped HAPI method uses the same generic type
    Optional<BaseRuntimeElementDefinition<?>> getElementDefinition(final Class<? extends IBase> theElementType);

    /**
     * Returns the scanned runtime model for the given type.
     *
     * @param theResourceName The name of the resource to load the resource definition for
     * @return the definition, if any
     * @see FhirContext#getResourceDefinition(String)
     */
    Optional<RuntimeResourceDefinition> getResourceDefinition(final String theResourceName);

    /**
     * Returns the scanned runtime model for the given type.
     *
     * @param theResource The resource to load the resource definition for
     * @return the definition, if any
     * @see FhirContext#getResourceDefinition(IBaseResource)
     */
    Optional<RuntimeResourceDefinition> getResourceDefinition(final IBaseResource theResource);
}
