/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.process;

import ca.uhn.fhir.context.BaseRuntimeElementDefinition;
import ca.uhn.fhir.context.FhirContext;
import ca.uhn.fhir.context.FhirVersionEnum;
import ca.uhn.fhir.context.RuntimeResourceDefinition;
import ca.uhn.fhir.parser.IParser;
import java.util.Optional;
import javax.inject.Named;
import javax.inject.Singleton;
import lombok.extern.slf4j.XSlf4j;
import org.hl7.fhir.instance.model.api.IBase;
import org.hl7.fhir.instance.model.api.IBaseResource;

/**
 * Default implementation of {@link IContextProvider}.
 */
@Named
@Singleton
@XSlf4j
public class ContextProvider implements IContextProvider {

    /** Create a new instance of ContextProvider */
    public ContextProvider() {
        // Default constructor with generated JavaDoc
    }

    private Optional<FhirVersionEnum> fhirVersion = Optional.empty();
    private Optional<FhirContext> fhirContext = Optional.empty();

    @Override
    public Optional<FhirVersionEnum> getFhirVersion() {
        return this.fhirVersion;
    }

    @Override
    public void setFhirVersion(FhirVersionEnum fhirVersion) throws IllegalStateException {
        logger.entry(fhirVersion);
        if (this.fhirContext.isPresent()) {
            if (!this.fhirVersion.orElseThrow().equals(fhirVersion)) {
                throw logger.throwing(new IllegalStateException(
                        "Changing the FHIR version after context instantiation is not allowed."));
            }
        } else {
            this.fhirVersion = Optional.of(fhirVersion);
        }
        logger.exit();
    }

    private FhirContext get() {
        logger.entry();
        if (this.fhirContext.isEmpty()) {
            if (this.fhirVersion.isEmpty()) {
                throw logger.throwing(
                        new IllegalStateException("Unable to instantiate the context without a FHIR version set."));
            }
            this.fhirContext = Optional.of(FhirContext.forVersion(this.fhirVersion.get()));
        }
        return logger.exit(this.fhirContext.get());
    }

    @Override
    public IParser newJsonParser() {
        return get().newJsonParser();
    }

    @Override
    public Optional<BaseRuntimeElementDefinition<?>> getElementDefinition(String theElementName) {
        logger.entry(theElementName);
        try {
            return logger.exit(Optional.ofNullable(get().getElementDefinition(theElementName)));
        } catch (final Exception e) {
            logger.debug("Unable to load element definition for {}", theElementName, e);
            return logger.exit(Optional.empty());
        }
    }

    @Override
    public Optional<BaseRuntimeElementDefinition<?>> getElementDefinition(Class<? extends IBase> theElementType) {
        logger.entry(theElementType);
        try {
            return logger.exit(Optional.ofNullable(get().getElementDefinition(theElementType)));
        } catch (final Exception e) {
            logger.debug("Unable to load element definition for {}", theElementType, e);
            return logger.exit(Optional.empty());
        }
    }

    @Override
    public Optional<RuntimeResourceDefinition> getResourceDefinition(String theResourceName) {
        logger.entry(theResourceName);
        try {
            return logger.exit(Optional.ofNullable(get().getResourceDefinition(theResourceName)));
        } catch (final Exception e) {
            logger.debug("Unable to load resource definition for {}: {}", theResourceName, e.getMessage());
            return logger.exit(Optional.empty());
        }
    }

    @Override
    public Optional<RuntimeResourceDefinition> getResourceDefinition(IBaseResource theResource) {
        logger.entry(theResource);
        try {
            return logger.exit(Optional.ofNullable(get().getResourceDefinition(theResource)));
        } catch (final Exception e) {
            logger.debug("Unable to load resource definition for {}: {}", theResource, e.getMessage());
            return logger.exit(Optional.empty());
        }
    }
}
