/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.packages;

import ca.uhn.fhir.parser.DataFormatException;
import ca.uhn.fhir.parser.IParser;
import ca.uhn.fhir.parser.LenientErrorHandler;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Multimap;
import com.google.common.collect.MultimapBuilder;
import de.fhlintstone.fhir.IResourceUtilities;
import de.fhlintstone.process.IContextProvider;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.inject.Named;
import javax.inject.Singleton;
import lombok.extern.slf4j.XSlf4j;
import org.apache.maven.artifact.versioning.ComparableVersion;
import org.hl7.fhir.instance.model.api.IBaseResource;
import org.hl7.fhir.utilities.json.model.JsonObject;
import org.hl7.fhir.utilities.npm.NpmPackage;
import org.hl7.fhir.utilities.npm.NpmPackage.NpmPackageFolder;

/**
 * Default implementation of {@link IPackageRegistry}.
 */
@Named
@Singleton
@XSlf4j
public class PackageRegistry implements IPackageRegistry {

    /**
     * Structure used to keep the information extracted for a resource during the
     * package scan process.
     *
     * @param resourceType the {@link FhirResourceType} of the resource
     * @param uri the canonical URI of the resource
     * @param version the version of the resource, if specified
     * @param fhirPackage the {@link IFhirPackage} that contains the resource
     * @param resource an {@link IBaseResource} object to allow for access to the resource
     */
    protected record ResourceInformation(
            FhirResourceType resourceType,
            URI uri,
            Optional<String> version,
            IFhirPackage fhirPackage,
            IBaseResource resource) {}

    private final IContextProvider contextProvider;
    private final IResourceUtilities resourceUtilities;

    final Collection<IFhirPackage> packages = new ArrayList<>();

    Multimap<IFhirPackage, ResourceInformation> resourcesByPackage =
            MultimapBuilder.hashKeys().linkedListValues().build();
    Multimap<URI, ResourceInformation> resourcesByURI =
            MultimapBuilder.hashKeys().linkedListValues().build();
    Multimap<FhirResourceType, ResourceInformation> resourcesByType =
            MultimapBuilder.hashKeys().linkedListValues().build();

    /**
     * Constructor for dependency injection
     *
     * @param contextProvider   the {@link IContextProvider} to use
     * @param resourceUtilities the {@link IResourceUtilities} to use
     */
    @Inject
    public PackageRegistry(IContextProvider contextProvider, IResourceUtilities resourceUtilities) {
        this.contextProvider = contextProvider;
        this.resourceUtilities = resourceUtilities;
    }

    @Override
    public void clear() {
        logger.entry();
        this.packages.clear();
        this.resourcesByPackage.clear();
        this.resourcesByURI.clear();
        this.resourcesByType.clear();
        logger.exit();
    }

    @Override
    public IFhirPackage register(File packageFile) throws PackageFileException {
        logger.entry(packageFile);

        // ensure that we're dealing with a readable file
        final String packageFileName = packageFile.getAbsolutePath();
        if (!packageFile.isFile() || !packageFile.canRead()) {
            throw new PackageFileException(
                    String.format("Package file %s is not a file or could not be read", packageFileName));
        }

        // load the package contents
        logger.debug("Attempting to load package file {}", packageFileName);
        try (var is = new FileInputStream(packageFile)) {
            // prepare access to contents and check folder structure
            final var npmPackage = NpmPackage.fromPackage(is);
            final var folders = npmPackage.getFolders();
            if (!folders.containsKey("package")) {
                throw new PackageFileException(
                        String.format("Package file %s does not contain a package folder", packageFileName));
            }

            // extract and validate manifest information
            final var fhirPackage = loadPackageHeader(packageFile, npmPackage);

            // check folder structure and read contents
            readPackageContents(fhirPackage, folders.get("package"));

            logger.info(
                    "Loaded package {} version {} from file {}",
                    fhirPackage.getName(),
                    fhirPackage.getVersion(),
                    packageFileName);
            return logger.exit(fhirPackage);
        } catch (final IOException | NullPointerException e) {
            throw logger.throwing(new PackageFileException(
                    String.format(
                            "Package file %s is not a valid .tar.gz archive, is missing the package folder or contains an invalid manifest",
                            packageFileName),
                    e));
        }
    }

    private IFhirPackage loadPackageHeader(File packageFile, NpmPackage npmPackage) throws PackageFileException {
        logger.entry(npmPackage);
        final var packageFileName = packageFile.getAbsolutePath();
        final var manifest = npmPackage.getNpm();
        final var name = readName(packageFileName, manifest);
        final var version = readVersion(packageFileName, manifest);
        final var canonicalURL = readCanonicalURL(packageFileName, manifest);
        final var dependencies = readDependencies(packageFileName, manifest);
        final var fhirPackage = FhirPackage.builder()
                .withSourceFile(packageFile)
                .withNpmPackage(npmPackage)
                .withName(name)
                .withVersion(version)
                .withCanonicalURL(canonicalURL)
                .withDependencies(dependencies)
                .build();
        this.packages.add(fhirPackage);
        return logger.exit(fhirPackage);
    }

    /**
     * Reads the name from the package manifest.
     *
     * @param packageFileName
     * @param manifest
     * @return the package name
     * @throws PackageFileException
     */
    private String readName(final String packageFileName, JsonObject manifest) throws PackageFileException {
        logger.entry(manifest);
        final var entries = manifest.getStrings("name");
        if (entries.size() != 1) {
            throw logger.throwing(new PackageFileException(
                    String.format("Manifest of package file %s contains an invalid name", packageFileName)));
        }
        return logger.exit(entries.getFirst());
    }

    /**
     * Reads the version from the package manifest.
     *
     * @param packageFileName
     * @param manifest
     * @return the package version
     * @throws PackageFileException
     */
    private String readVersion(final String packageFileName, JsonObject manifest) throws PackageFileException {
        logger.entry(manifest);
        final var entries = manifest.getStrings("version");
        if (entries.size() != 1) {
            throw new PackageFileException(
                    String.format("Manifest of package file %s contains an invalid version", packageFileName));
        }
        return logger.exit(entries.getFirst());
    }

    /**
     * Reads the canonical URL from the package manifest.
     *
     * @param packageFileName
     * @param manifest
     * @return the canonical URL, if set
     * @throws PackageFileException
     */
    private Optional<URL> readCanonicalURL(final String packageFileName, JsonObject manifest)
            throws PackageFileException {
        logger.entry(manifest);
        final var entries = manifest.getStrings("canonical");
        switch (entries.size()) {
            case 0:
                logger.debug("Manifest of package file {} does not contain a canonical URL", packageFileName);
                return logger.exit(Optional.empty());
            case 1:
                final var canonical = entries.getFirst();
                try {
                    final var canonicalURL = new URI(canonical).toURL();
                    return logger.exit(Optional.of(canonicalURL));
                } catch (final MalformedURLException | URISyntaxException e) {
                    throw new PackageFileException(String.format(
                            "Manifest of package file %s contains an invalid canonical URL %s",
                            packageFileName, canonical));
                }
            default:
                throw new PackageFileException(String.format(
                        "Manifest of package file %s contains an invalid canonical URL", packageFileName));
        }
    }

    /**
     * Reads the dependencies from the package manifest.
     *
     * @param packageFileName
     * @param manifest
     * @return the dependencies (list may be empty in special cases)
     * @throws PackageFileException
     */
    private Map<String, String> readDependencies(String packageFileName, JsonObject manifest)
            throws PackageFileException {
        logger.entry(manifest);
        final var result = new HashMap<String, String>();

        // dependencies entry must exist and be an object
        // see https://hl7.org/fhir/packages.html#2.1.10.2: "A dependency on a core
        // package is required"
        // HOWEVER: The core packages themselves do not declare any dependencies, so
        // make this a warning only.
        final var deps = manifest.get("dependencies");
        if ((deps == null) || !deps.isJsonObject()) {
            // TODO #8 check whether this is a core package
            // TODO #8 add a unit test to cover this case
            logger.warn("Manifest file of package file {} does not contain any dependencies.", packageFileName);
        } else {
            for (final var property : deps.asJsonObject().getProperties()) {
                final var dependencyName = property.getName();
                final var dependencyVersion = property.getValue();
                if (!dependencyVersion.isJsonString()) {
                    logger.error(
                            "Version of dependency {} in package {} is not a string", dependencyName, packageFileName);
                    throw new PackageFileException(String.format(
                            "Manifest file of package file %s contains an invalid dependency.", packageFileName));
                }
                result.put(dependencyName, dependencyVersion.asJsonString().getValue());
            }
        }
        return logger.exit(result);
    }

    /**
     * Reads the package contents and adds the resources to the index
     *
     * @param fhirPackage
     * @param packageFolder
     */
    private void readPackageContents(IFhirPackage fhirPackage, final NpmPackageFolder packageFolder) {
        var ignoredResources = 0;
        for (final String nextFile : packageFolder.listFiles()) {
            if (nextFile.toLowerCase(Locale.US).endsWith(".json")) {
                final String input = new String(packageFolder.getContent().get(nextFile), StandardCharsets.UTF_8);
                final IParser parser = this.contextProvider.newJsonParser();
                parser.setParserErrorHandler(new LenientErrorHandler(false));
                IBaseResource resource;
                try {
                    resource = parser.parseResource(input);
                    final var resourceType = readResourceType(resource, nextFile);
                    if (resourceType.isPresent()) {
                        final var resourceURI = this.resourceUtilities.readResourceURI(resource);
                        final Optional<URI> uri = resourceURI.uri();
                        if (uri.isEmpty()) {
                            logger.debug(
                                    "Unable to determine URI of resource contained in {}, file is ignored", nextFile);
                        } else {
                            final var info = new ResourceInformation(
                                    resourceType.get(), uri.get(), resourceURI.version(), fhirPackage, resource);
                            this.resourcesByPackage.put(fhirPackage, info);
                            this.resourcesByURI.put(info.uri, info);
                            this.resourcesByType.put(resourceType.get(), info);
                        }
                    } else {
                        ignoredResources++;
                    }
                } catch (final DataFormatException e) {
                    logger.debug("Unable to parse resource contained in file {}, file is ignored", nextFile, e);
                    ignoredResources++;
                }
            }
        }
        if (ignoredResources > 0) {
            logger.warn(
                    "{} resources in package {} were ignored because the resource could not be parsed or the resource type could not determined. For more details, check the debug log.",
                    ignoredResources,
                    fhirPackage.getName());
        }
    }

    /**
     * Reads the FHIR type of a resource.
     *
     * @param resource
     * @param fileName
     * @return
     */
    private Optional<FhirResourceType> readResourceType(IBaseResource resource, String fileName) {
        logger.entry(resource, fileName);
        try {
            return logger.exit(Optional.of(FhirResourceType.fromResource(resource)));
        } catch (final IllegalArgumentException e) {
            logger.debug("Unable to determine type of resource contained in {}, file is ignored", fileName, e);
            return logger.exit(Optional.empty());
        }
    }

    @Override
    public ImmutableList<IBaseResource> getResources(IFhirPackage fhirPackage) {
        logger.entry(fhirPackage);
        if (!this.resourcesByPackage.containsKey(fhirPackage)) {
            throw logger.throwing(new IllegalArgumentException(
                    String.format("Package %s has not been registered", fhirPackage.getName())));
        }
        final var result = this.resourcesByPackage.get(fhirPackage).stream()
                .map(info -> info.resource)
                .toList();
        return logger.exit(ImmutableList.copyOf(result));
    }

    @Override
    public ImmutableList<IBaseResource> getResources(FhirResourceType resourceType, IFhirPackage fhirPackage) {
        logger.entry(resourceType, fhirPackage);
        if (!this.resourcesByPackage.containsKey(fhirPackage)) {
            throw logger.throwing(new IllegalArgumentException(
                    String.format("Package %s has not been registered", fhirPackage.getName())));
        }
        final var result = this.resourcesByPackage.get(fhirPackage).stream()
                .filter(info -> info.resourceType.equals(resourceType))
                .map(info -> info.resource)
                .toList();
        return logger.exit(ImmutableList.copyOf(result));
    }

    @Override
    public ImmutableList<IBaseResource> getResources(URI resourceURI) {
        logger.entry(resourceURI);
        final var result = this.resourcesByURI.get(resourceURI).stream()
                .map(info -> info.resource)
                .toList();
        return logger.exit(ImmutableList.copyOf(result));
    }

    @Override
    public ImmutableList<IBaseResource> getResources(FhirResourceType resourceType, URI resourceURI) {
        logger.entry(resourceType, resourceURI);
        final var result = this.resourcesByURI.get(resourceURI).stream()
                .filter(info -> info.resourceType.equals(resourceType))
                .map(info -> info.resource)
                .toList();
        return logger.exit(ImmutableList.copyOf(result));
    }

    @Override
    public Optional<IBaseResource> getUniqueResource(URI resourceURI) {
        return getUniqueResource(null, resourceURI);
    }

    @Override
    public Optional<IBaseResource> getUniqueResource(FhirResourceType resourceType, URI resourceURI) {
        logger.entry(resourceURI);
        final var matchingResources = this.resourcesByURI.get(resourceURI);
        final var result = matchingResources.stream()
                .filter(info -> {
                    if (resourceType == null) {
                        return true;
                    }
                    return info.resourceType == resourceType;
                })
                .max(ResourceVersionComparator::compare);

        if (result.isEmpty()) {
            return logger.exit(Optional.empty());
        }
        return logger.exit(Optional.of(result.get().resource));
    }

    @Override
    public ImmutableList<String> getUnmetPackageDependencies() {
        logger.entry();

        // NPM package.json allows for arbitrary whitespace around the version. No idea
        // why anyone would consider that a sensible idea...
        // Also, according to FHIR, all package names should be lower case, but some
        // people just can't bother...

        // TODO #10 add unit tests for these edge cases as well

        final Multimap<String, String> versionsAvailable =
                MultimapBuilder.hashKeys().hashSetValues().build();
        for (final var pkg : this.packages) {
            versionsAvailable.put(pkg.getName().toLowerCase(), pkg.getVersion().trim());
            // allow for n.n.X dependencies (ignoring patch level)
            final var patchWildcardVersion = pkg.getVersion().replaceAll("\\.\\d+$", ".x");
            versionsAvailable.put(pkg.getName(), patchWildcardVersion);
        }

        final var result = new ArrayList<String>();
        for (final var pkg : this.packages) {
            for (final var dep : pkg.getDependencies().entrySet()) {
                final var depName = dep.getKey().toLowerCase();
                final var depVersion = dep.getValue().trim();
                if (!versionsAvailable.get(depName).contains(depVersion)) {
                    result.add(depName + "#" + depVersion);
                }
            }
        }

        return logger.exit(ImmutableList.copyOf(result));
    }

    @Override
    public Optional<IFhirPackage> getPackageOfResource(URI resourceURI) throws AmbiguousResourceURIException {
        logger.entry(resourceURI);
        Optional<IFhirPackage> result = Optional.empty();
        final var matchingResources = this.resourcesByURI.get(resourceURI);
        final var info =
                matchingResources.stream().map(entry -> entry.fhirPackage).collect(Collectors.toSet());
        if (info.size() > 1) {
            final var sourcePackages = matchingResources.stream()
                    .map(ResourceInformation::fhirPackage)
                    .toList();
            throw logger.throwing(new AmbiguousResourceURIException(resourceURI, sourcePackages));
        } else if (info.size() == 1) {
            result = info.stream().findFirst();
        }
        return logger.exit(result);
    }

    /**
     * Compares two ResourceInformation objects based on their version numbers.
     *
     * @see #compare(ResourceInformation, ResourceInformation)
     */
    protected static final class ResourceVersionComparator {

        private ResourceVersionComparator() {}

        /**
         * Compares two ResourceInformation objects based on their version numbers.
         *
         * This method implements version-based comparison with the following precedence:<br>
         * 1. If both resources have the same version (including both empty), they are considered equal<br>
         * 2. Resources with empty versions are considered "less than" resources with versions<br>
         * 3. Resources with versions are compared using semantic version comparison via {@link ComparableVersion}
         *
         * @param resource1 the first ResourceInformation object to compare
         * @param resource2 the second ResourceInformation object to compare
         * @return a negative integer if resource1.version &lt; resource2.version,
         *         zero if resource1.version = resource2.version,
         *         or a positive integer if resource1.version &gt; resource2.version
         *
         * @see ComparableVersion
         */
        public static int compare(ResourceInformation resource1, ResourceInformation resource2) {
            logger.entry(resource1, resource2);
            if (resource1.version.equals(resource2.version)) {
                return logger.exit(0);
            }
            if (resource1.version.isEmpty()) {
                return logger.exit(-1);
            }
            if (resource2.version.isEmpty()) {
                return logger.exit(1);
            }
            final var resource1Version = new ComparableVersion(resource1.version.get());
            final var resource2Version = new ComparableVersion(resource2.version.get());
            return logger.exit(resource1Version.compareTo(resource2Version));
        }
    }
}
