/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.packages;

import com.google.common.collect.ImmutableList;
import java.io.File;
import java.net.URI;
import java.util.Optional;
import org.hl7.fhir.instance.model.api.IBaseResource;

/**
 * Keeps track of the FHIR packages and their contents.
 *
 * @see <a href="https://hl7.org/fhir/packages.html">FHIR Packages</a>
 */
public interface IPackageRegistry {

    /**
     * Removes all registered packages from the registry and returns it to its initial state.
     */
    void clear();

    /**
     * Adds a package (contained in a NPM package file) to the registry.
     *
     * @param packageFile
     * @return the package object
     * @throws PackageFileException
     */
    IFhirPackage register(File packageFile) throws PackageFileException;

    /**
     * Retrieves a list of all resources accessible from a package.
     *
     * @param fhirPackage the {@link IFhirPackage} object
     * @return the list of all resources accessible from the package
     */
    ImmutableList<IBaseResource> getResources(IFhirPackage fhirPackage);

    /**
     * Retrieves a list of all resources of a certain type accessible from a
     * package.
     *
     * @param resourceType the {@link FhirResourceType} to filter for
     * @param fhirPackage  the {@link IFhirPackage} object
     * @return the list of all resources of the given type accessible from the
     *         package
     */
    ImmutableList<IBaseResource> getResources(FhirResourceType resourceType, IFhirPackage fhirPackage);

    /**
     * Retrieves a list of all resources designated with a certain URI.
     *
     * @param resourceURI the {@link URI} to filter for
     * @return the list of all resources claiming the URI given
     */
    ImmutableList<IBaseResource> getResources(URI resourceURI);

    /**
     * Retrieves a list of all resources of a certain type designated with a certain
     * URI.
     *
     * @param resourceType the {@link FhirResourceType} to filter for
     * @param resourceURI  the {@link URI} to filter for
     * @return the list of all resources of the given type claiming the URI given
     */
    ImmutableList<IBaseResource> getResources(FhirResourceType resourceType, URI resourceURI);

    /**
     * Retrieves a unique resource designated with a certain URI, if present.
     *
     * @param resourceURI the {@link URI} to filter for
     * @return an {@link Optional} of the resource (which may be empty if no
     *         matching resource was found). If multiple resources match the criteria,
     *         the one with the highest version is returned.
     */
    Optional<IBaseResource> getUniqueResource(URI resourceURI);

    /**
     * Retrieves a unique resource of a certain type designated with a certain URI,
     * if present.
     *
     * @param resourceType the {@link FhirResourceType} to filter for
     * @param resourceURI  the {@link URI} to filter for
     * @return an {@link Optional} of the resource (which may be empty if no
     *         matching resource was found). If multiple resources match the criteria,
     *         the one with the highest version is returned.
     */
    Optional<IBaseResource> getUniqueResource(FhirResourceType resourceType, URI resourceURI);

    /**
     * Returns a list of all package dependencies that are not yet resolvable by the
     * registry.
     *
     * @return a list of all package dependencies that are not yet resolvable by the
     *         registry
     */
    ImmutableList<String> getUnmetPackageDependencies();

    /**
     * Determines the package that contains a resource.
     *
     * @param resourceURI the resource URI to search for
     * @return the {@link IFhirPackage} that contains the resource
     * @throws AmbiguousResourceURIException
     */
    // TODO: Soll das die höchste Version zurückgeben?
    Optional<IFhirPackage> getPackageOfResource(URI resourceURI) throws AmbiguousResourceURIException;
}
