/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.packages;

import com.google.common.collect.ImmutableMap;
import java.io.File;
import java.net.URL;
import java.util.Optional;

/**
 * Representation of a FHIR package read from an input file.
 * @see <a href="https://hl7.org/fhir/packages.html">FHIR Packages</a>
 */
public interface IFhirPackage extends IFhirPackageBase {

    /**
     * Returns the source file the package information was read from.
     * @return the source file the package information was read from
     */
    File getSourceFile();

    /**
     * Returns the package name without version information.
     * This information is contained in the mandatory name attribute of the package manifest.
     * @return the package name without version information
     */
    String getName();

    /**
     * Returns the package version.
     * This information is contained in the mandatory version attribute of the package manifest.
     * @return the package version
     */
    String getVersion();

    /**
     * Returns the canonical package URL.
     * This information is contained in the mandatory canonical attribute of the package manifest.
     * @return the canonical package URL
     */
    Optional<URL> getCanonicalURL();

    /**
     * Returns the dependency information as a map where the key is the package name or the local path and the value
     * contains the version information.
     *
     * A dependency on a core package is required, and this dependency defines the basic
     *
     * Development packages may use a local path for the package version (see Local Paths icon in the NPM documentation).
     *
     * Packages SHOULD be referred to by the whole package version number and not use wildcards, except for the patch
     * version in a semver version reference. The x here (e.g. in 5.0.x) means that a package resolver should accept
     * the package with the highest found patch number.
     * @return the dependency information
     */
    ImmutableMap<String, String> getDependencies();
}
