/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.packages;

import com.google.common.base.Strings;
import org.hl7.fhir.instance.model.api.IBaseResource;

/**
 * These values represent the known values returned by {@link IBaseResource#fhirType()}
 */
public enum FhirResourceType {

    /**
     * A Capability Statement documents a set of capabilities (behaviors) of a FHIR Server or Client for a particular
     * version of FHIR that may be used as a statement of actual server functionality or a statement of
     * required or desired server implementation.
     *
     * @see <a href="https://www.hl7.org/fhir/capabilitystatement.html">CapabilityStatement</a>
     */
    CAPABILITY_STATEMENT,

    /**
     * The CodeSystem resource is used to declare the existence of and describe a code system or code system
     * supplement and its key properties, and optionally define a part or all of its content.
     *
     * @see <a href="https://www.hl7.org/fhir/codesystem.html">CodeSystem</a>
     */
    CODE_SYSTEM,

    /**
     * A compartment definition that defines how resources are accessed on a server.
     *
     * @see <a href="https://hl7.org/fhir/compartmentdefinition.html">CompartmentDefinition</a>
     */
    COMPARTMENT_DEFINITION,

    /**
     * A statement of relationships from one set of concepts to one or more other concepts - either concepts in
     * code systems, or data element/data element concepts, or classes in class models.
     *
     * @see <a href="https://www.hl7.org/fhir/conceptmap.html">ConceptMap</a>
     */
    CONCEPT_MAP,

    /**
     * A set of rules of how a particular interoperability or standards problem is solved - typically through the use of FHIR resources. This resource is used to gather all the parts of an implementation guide into a logical whole and to publish a computable definition of all the parts.
     *
     * @see <a href="https://www.hl7.org/fhir/implementationguide.html">ImplementationGuide</a>
     */
    IMPLEMENTATION_GUIDE,

    /**
     * A curated namespace that issues unique symbols within that namespace for the identification of concepts, people, devices, etc. Represents a "System" used within the Identifier and Coding data types.
     *
     * @see <a href="https://hl7.org/fhir/namingsystem.html">NamingSystem</a>
     */
    NAMING_SYSTEM,

    /**
     * A formal computable definition of an operation (on the RESTful interface) or a named query (using the search interaction).
     *
     * @see <a href="https://www.hl7.org/fhir/operationdefinition.html">OperationDefinition</a>
     */
    OPERATION_DEFINITION,

    /**
     * A search parameter that defines a named search item that can be used to search/filter on a resource.
     *
     * @see <a href="https://www.hl7.org/fhir/searchparameter.html">SearchParameter</a>
     */
    SEARCH_PARAMETER,

    /**
     * A definition of a FHIR structure. This resource is used to describe the underlying resources, data types defined in FHIR, and also for describing extensions and constraints on resources and data types.
     *
     * @see <a href="https://www.hl7.org/fhir/structuredefinition.html">StructureDefinition</a>
     */
    STRUCTURE_DEFINITION,

    /**
     * A Map of relationships between 2 structures that can be used to transform data.
     *
     * @see <a href="https://www.hl7.org/fhir/structuremap.html">StructureMap</a>
     */
    STRUCTURE_MAP,

    /**
     * A ValueSet resource instance specifies a set of codes drawn from one or more code systems, intended for use in a particular context. Value sets link between CodeSystem definitions and their use in coded elements.
     *
     * @see <a href="https://www.hl7.org/fhir/valueset.html">ValueSet</a>
     */
    VALUE_SET;

    /**
     * Determines the enum value from the string representation
     * @param value the string representation of the resource type
     * @return the corresponding enum representation
     */
    public static FhirResourceType fromString(String value) {
        if (value.equals("CapabilityStatement")) {
            return CAPABILITY_STATEMENT;
        } else if (value.equals("CodeSystem")) {
            return CODE_SYSTEM;
        } else if (value.equals("CompartmentDefinition")) {
            return COMPARTMENT_DEFINITION;
        } else if (value.equals("ConceptMap")) {
            return CONCEPT_MAP;
        } else if (value.equals("ImplementationGuide")) {
            return IMPLEMENTATION_GUIDE;
        } else if (value.equals("NamingSystem")) {
            return NAMING_SYSTEM;
        } else if (value.equals("OperationDefinition")) {
            return OPERATION_DEFINITION;
        } else if (value.equals("SearchParameter")) {
            return SEARCH_PARAMETER;
        } else if (value.equals("StructureDefinition")) {
            return STRUCTURE_DEFINITION;
        } else if (value.equals("StructureMap")) {
            return STRUCTURE_MAP;
        } else if (value.equals("ValueSet")) {
            return VALUE_SET;
        } else {
            throw new IllegalArgumentException(String.format("Unknown FHIR resource type %s", value));
        }
    }

    /**
     * Determines the enum value from an {@link IBaseResource}
     * @param resource the {@link IBaseResource} to examine
     * @return the corresponding enum representation
     */
    public static FhirResourceType fromResource(IBaseResource resource) {
        final var typeString = resource.fhirType();
        if (Strings.isNullOrEmpty(typeString)) {
            throw new IllegalArgumentException("Resource did not provide a FHIR type");
        }
        return FhirResourceType.fromString(typeString);
    }

    @Override
    public String toString() {
        switch (this) {
            case CAPABILITY_STATEMENT:
                return "CapabilityStatement";
            case CODE_SYSTEM:
                return "CodeSystem";
            case COMPARTMENT_DEFINITION:
                return "CompartmentDefinition";
            case CONCEPT_MAP:
                return "ConceptMap";
            case IMPLEMENTATION_GUIDE:
                return "ImplementationGuide";
            case NAMING_SYSTEM:
                return "NamingSystem";
            case OPERATION_DEFINITION:
                return "OperationDefinition";
            case SEARCH_PARAMETER:
                return "SearchParameter";
            case STRUCTURE_DEFINITION:
                return "StructureDefinition";
            case STRUCTURE_MAP:
                return "StructureMap";
            case VALUE_SET:
                return "ValueSet";
            default:
                throw new IllegalArgumentException(String.format("Unknown FHIR resource type %s", this));
        }
    }
}
