/*
 *
 * Fhlintstone FHIR implementation generator
 *
 * Copyright (C) 2025 Fhlintstone authors and contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package de.fhlintstone.packages;

import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableList;
import java.net.URI;
import java.util.Collection;
import java.util.stream.Collectors;
import lombok.Getter;

/**
 * This exception is thrown if the {@link IPackageRegistry} is queried for a resource URI that is supposed to be
 * unique, but multiple resources are found claiming that URI.
 */
public class AmbiguousResourceURIException extends Exception {

    @Getter
    private final URI uri;

    @Getter
    private final ImmutableCollection<IFhirPackage> containingPackages;

    /**
     * Creates a new exception for the given URI.
     * @param uri the URI that has multiple resources assigned to it
     * @param containingPackages the packages that contain the conflicting resources
     */
    public AmbiguousResourceURIException(URI uri, Collection<IFhirPackage> containingPackages) {
        super(String.format(
                "The URI %s is not unique within the loaded packages (found in the following packages: %s)",
                uri, containingPackages.stream().map(p -> p.toString()).collect(Collectors.joining(", "))));
        this.uri = uri;
        this.containingPackages = ImmutableList.copyOf(containingPackages);
    }
}
